/**
  *  \file vmalloc.h
  */
#ifndef ILD2_VMALLOC_H
#define ILD2_VMALLOC_H

#include <list>
#include <map>
#include <vector>
#include "cpluslib/string.h"
#include "vfile.h"
#include "cpluslib/smartptr.h"
#include "section.h"
#include "got.h"

class ScriptParser;
class Output;
class StateFileWriter;

extern const char got_sym_name[], dyn_sym_name[];

class MemoryAllocator {
 public:
    MemoryAllocator() { }
    virtual ~MemoryAllocator() { }

    /* Create an `LinkObject' and add it to this memory allocator. */
    virtual void addObject(string_t name, Ptr<VFile> file) = 0;

    /* Add a section. Called by Section->getInformation */
    virtual void addSection(Ptr<ProgramSection> sec) = 0;

    /* Add a common symbol */
    virtual void addCommon(string_t name, Elf32_Word size,
                           Elf32_Word align) = 0;

    /* Force creation of a GOT */
    virtual void createGot() = 0;

    /* Get offset of a symbol in the GOT */
    virtual Elf32_Word getGotOffset(string_t name) = 0;

    /* Compute addresses, sections, ... */
    virtual void computeAddresses(ScriptParser& parser) = 0;

    /* Write output (open, write, close) */
    virtual void writeOutput(Output&) = 0;

    /* Write state file */
    virtual void writeStateFile(StateFileWriter&) = 0;
};

class DefaultMemoryAllocator : public MemoryAllocator {
    /* Definition of a common symbol */
    struct CommonInfo {
        Elf32_Word size;
        Elf32_Word align;
        CommonInfo(Elf32_Word s, Elf32_Word a)
            : size(s), align(a) { }
        CommonInfo()            // needed by std::map
            { }
    };

    /* List of sections */
    typedef std::list<Ptr<ProgramSection> > list_type;
    /* For each section name, all sections */
    typedef std::map<string_t, list_type> map_type;
    /* Vector of objects */
    typedef std::vector<Ptr<LinkObject> > objvec_type;
    /* Map of common symbols */
    typedef std::map<string_t, CommonInfo> common_type;
    /* Padding data */
    typedef std::map<Elf32_Word, Elf32_Word> padmap_type;

    map_type        sections;
    objvec_type     objects;
    common_type     common_symbols;
    padmap_type     padding;

    Ptr<ProgbitsSection>   common_section;
    Ptr<GlobalOffsetTable> got_section;

    Elf32_Word input_add,  input_align;
    Elf32_Word output_add, output_align;

    Elf32_Addr first_address;

    /* called by computeAddresses */
    void buildCommonSection();
    void addPadding(Elf32_Addr& adr, Elf32_Word add, Elf32_Word align);
    void addSections(string_t name, string_t file, Elf32_Addr& adr, Elf32_Addr rel_diff);
 public:
    DefaultMemoryAllocator();
    ~DefaultMemoryAllocator();

    void addObject(string_t name, Ptr<VFile> file);
    void addSection(Ptr<ProgramSection> sec);
    void addCommon(string_t name, Elf32_Word size, Elf32_Word align);

    void createGot();
    Elf32_Word getGotOffset(string_t name);

    void computeAddresses(ScriptParser& parser);

    void writeOutput(Output& output);
    void writeStateFile(StateFileWriter&);
};

#endif
