/**
  *  \file vmalloc.cc
  *  \brief Virtual Memory Allocator (linkee image)
  */

#include "vmalloc.h"
#include "symbol.h"
#include "object.h"
#include "cpluslib/bitops.h"
#include "log.h"
#include "script.h"
#include "cpluslib/assert.h"
#include "address.h"
#include "wildcard.h"
#include "namemap.h"
#include "except.h"
#include "output.h"
#include "param.h"
#include "state.h"

/* Symbol names */
const char got_sym_name[] = "_GLOBAL_OFFSET_TABLE_";
const char dyn_sym_name[] = "_DYNAMIC_";
const char dummy1_name[] = "__ild_got_dummy1";
const char dummy2_name[] = "__ild_got_dummy2";


static const Elf32_Addr NO_ADDRESS = ~0U;

/// Header used for common section.
static Elf32_Shdr common_header = {
    0,                          // sh_name
    SHT_NOBITS,                 // sh_type
    SHF_WRITE | SHF_ALLOC,      // sh_flags
    0,                          // sh_addr
    0,                          // sh_offset
    0,                          // sh_size (will be modified in common section)
    0,                          // sh_link
    0,                          // sh_info
    1,                          // sh_addralign -- FIXME?
    0                           // sh_entsize
};

/************************* DefaultMemoryAllocator ************************/

DefaultMemoryAllocator::DefaultMemoryAllocator()
    : sections(), objects(), common_symbols(),
      common_section(new ProgbitsSection(0, common_header)),
      first_address(NO_ADDRESS)
{ }

DefaultMemoryAllocator::~DefaultMemoryAllocator()
{ }

void
DefaultMemoryAllocator::addObject(string_t name, Ptr<VFile> f)
{
    Ptr<ElfObject> o = new ElfObject(f, name);
    if (o->getType() != ET_REL)
        throw LinkerError("File `" + name + "' is not relocatable file");
    objects.push_back(o);
    o->getInformation(symbols, *this);
}

void
DefaultMemoryAllocator::addSection(Ptr<ProgramSection> sec)
{
    sections[sec->getName()].push_back(sec);
}

void
DefaultMemoryAllocator::addCommon(string_t name, Elf32_Word size,
                                  Elf32_Word align)
{
    common_type::iterator i = common_symbols.find(name);
    if(i != common_symbols.end()) {
        if(i->second.size != size || i->second.align != align)
            log(LOG_ERROR) << "Common symbol `" << name << "' definition changed: "
                           << i->second.size << "/" << i->second.align
                           << " vs. "
                           << size << "/" << align << std::endl;
    } else {
        if(!isPowerOfTwo(align)) {
            log(LOG_ERROR) << "Common symbol `" << name << "' not aligned to power of two"
                           << std::endl;
            int i = 0;
            while(align) {
                align >>= 1;
                ++i;
            }
            align = 1 << i;
        }
        common_symbols[name] = CommonInfo(size, align);
        symbols.addSymbol(name, false /* not weak */, true /* defined */,
                          common_section, 0, 0);
    }
}

void
DefaultMemoryAllocator::buildCommonSection()
{
    Elf32_Word min_align = 1;
    Elf32_Word address = 0;

    /* compute all symbol addresses */
    for(common_type::iterator i = common_symbols.begin(); i != common_symbols.end(); ++i)
    {
        /* alignment */
        ASSERT(isPowerOfTwo(i->second.align));
        if(i->second.align > min_align)
            min_align = i->second.align;
        address = roundUp(address, i->second.align);
        /* fix symbol. Because symbols are defined non-weak, we can
           be sure we get the right symbol definition here. */
        Ptr<Symbol> sym = symbols.getSymbol(i->first);
        ASSERT(sym.ptr());
        ASSERT(sym->section == common_section);
        sym->address = address;
        address += i->second.size;
    }

    /* fix common section */
    if(address) {
        common_section->setName(".bss");
        common_section->setSize(address);
        common_section->setAlignment(min_align);
        addSection(common_section);
        log(LOG_DEBUG) << "Common section is " << address << " bytes, aligned " << min_align << "\n";
    } else {
        log(LOG_DEBUG) << "No common section needed.\n";
    }
}

void
DefaultMemoryAllocator::createGot()
{
    if (got_section)            // GOT already exists
        return;

    /* create GOT */
    got_section = new GlobalOffsetTable();
    symbols.addSymbol(got_sym_name, false /* weak */, true /* defined */,
                      got_section, 0 /* address */, 0 /* object */);
    if (dynamic_link) {
        /* dynamically linking: add symbols as mandated by the ELF standard */
        getGotOffset(dyn_sym_name);     // _DYNAMIC_
        getGotOffset(dummy1_name);      // entry used by ld.so
        getGotOffset(dummy2_name);      // entry used by ld.so
        ASSERT(getGotOffset(dyn_sym_name) == 0);
    } else {
        /* we're not linking dynamically. Since this violates the ELF
           standard anyway (ELF programs should import syscalls from
           libc.so), and we don't have a dynamic linker, we need not care
           about the linker symbols. */
        getGotOffset(dyn_sym_name);     // _DYNAMIC_
    }
}

Elf32_Word
DefaultMemoryAllocator::getGotOffset(string_t name)
{
    /* ensure symbol is available */
    symbols.addSymbol(name, true /* weak */, false /* defined */, 0, 0, 0);
    Ptr<Symbol> sym = symbols.getSymbol(name);
    ASSERT(sym.ptr());
    if(sym->got_offset != NO_TABLE_ADDRESS)
        return sym->got_offset;

    /* need to create a GOT entry */
    if(!got_section.ptr())
        createGot();
    
    return sym->got_offset = got_section->addEntry(name);
}

void
DefaultMemoryAllocator::computeAddresses(ScriptParser& script)
{
    buildCommonSection();
    if (got_section)
        addSection(got_section);

    Elf32_Addr link_addr = 0, rel_diff = 0;
    string_t   prev_section;

    input_add    = 0;
    input_align  = 1;
    output_add   = 0;
    output_align = PAGE_SIZE;

    for (ScriptParser::iterator i = script.begin(); i != script.end(); ++i) {
        switch(i->action) {
         case ScriptCommand::a_NOP:
            break;
         case ScriptCommand::a_DefineSymbol:
            log(LOG_ERROR) << "Unsupported: a_DefineSymbol\n";
            break;
         case ScriptCommand::a_SetLoadAddress:
            {
                Elf32_Addr new_addr = i->ex1->evalAbsolute();
                if (new_addr < link_addr)
                    log(LOG_WARN) << "Attempt to lower link address has been ignored.\n";
                else
                    link_addr = new_addr;
                rel_diff = 0;
            }
            break;
         case ScriptCommand::a_SetRelocationAddress:
            rel_diff = i->ex1->evalAbsolute() - link_addr;
            break;
         case ScriptCommand::a_StartSection:
            /* Start a new section */
            if (prev_section.length())
                symbols.addSymbol("__stop_" + prev_section, true, true, 0, link_addr + rel_diff, 0);
            addPadding(link_addr, prev_section.length() ? output_add : 0, output_align);
            log(LOG_DEBUG) << "section " << i->name << ": " << itox(link_addr) << "\n";
            symbols.addSymbol("__start_" + i->name, true, true, 0, link_addr + rel_diff, 0);
            prev_section = i->name;
            break;
         case ScriptCommand::a_AddSection:
            /* add section to image */
            addSections(i->name, i->file, link_addr, rel_diff);
            break;
         case ScriptCommand::a_SetOutputPad:
            output_add   = i->ex1->evalAbsolute();
            output_align = i->ex2->evalAbsolute();
            break;
         case ScriptCommand::a_SetInputPad:
            input_add   = i->ex1->evalAbsolute();
            input_align = i->ex2->evalAbsolute();
            break;
         case ScriptCommand::a_SetInputPadPattern:
            log(LOG_ERROR) << "unsupported: a_SetInputPadPattern\n";
            break;
        }
    }

    if(prev_section.length())
        symbols.addSymbol("__stop_" + prev_section, true, true, 0, link_addr + rel_diff, 0);
}

void
DefaultMemoryAllocator::addPadding(Elf32_Addr& adr, Elf32_Word add, Elf32_Word align)
{
    Elf32_Word end = roundUp(adr + add, align);

    if(end != adr) {
        padmap_type::iterator r = padding.lower_bound(adr);
        if(r == padding.end() || r->second < adr) {
            padding[adr] = end;
        } else {
            /* would be overlapping output addresses which can't happen */
            ASSERT(r->second == adr);
            r->second = end;
        }
    }
    adr = end;
}

// Add all sections matching `filewild(namewild)' to image.
void
DefaultMemoryAllocator::addSections(string_t name, string_t file, Elf32_Addr& adr, Elf32_Addr rel_diff)
{
    Wildcard namewild, filewild;

    /* script reader must make sure these wildcards are valid */
    ASSERT(namewild.setExpr(name) && filewild.setExpr(file));

    map_type::iterator beg = sections.lower_bound(namewild.getPrefix());
    map_type::iterator end = sections.end(); // upper_bound(namewild.getPrefix());

    for(map_type::iterator i = beg; i != end; ++i) {
        /* match section name */
        if (!namewild.match(i->first))
            continue;
        /* okay, we have a list of sections with matching names.
           Now check file names */
        list_type& s = i->second;
        for(list_type::iterator it = s.begin(); it != s.end(); ++it) {
            LinkObject* o = (*it)->getObject();
            string_t fname = o ? o->getName() : string_t("<unknown>");
            if(filewild.match(fname) && !(*it)->isVisited()) {
                /* we want this section. So be it. Compute start address. */
                Elf32_Word align = (*it)->getAlignment();
                if(align == 0)
                    align = 1;
                if(align & (align-1))
                    throw LinkerError(Section::getSectionName(*it), "invalid alignment request (only powers of 2 allowed)");

                /* padding before */
                addPadding(adr, 0, align);

                /* add the section */
                if (first_address == NO_ADDRESS)
                    first_address = adr;
                (*it)->setAddress(adr);
                (*it)->setRelocationAddress(adr + rel_diff);
                adr += (*it)->getSize();
                (*it)->markVisited();

                /* padding after */
                addPadding(adr, input_add, input_align);

                /* this was a linkonce section, so don't bother searching
                   more matching sections. Users can explicitly link in
                   particular incarnations by using appropriate wildcards. */
                if((*it)->isLinkOnce())
                    break;
            }
        }
    }
}

void
DefaultMemoryAllocator::writeOutput(Output& output)
{
    if (first_address == NO_ADDRESS)
        throw LinkerError("Image would be empty.");
    output.startOutput(symbols.getAddressOf(start_symbol), first_address);

    for(map_type::iterator i = sections.begin(); i != sections.end(); ++i)
        for(list_type::iterator j = i->second.begin(); j != i->second.end(); ++j)
            if((*j)->isInImage())
                (*j)->writeOutput(output);

    output.endOutput();
}

void
DefaultMemoryAllocator::writeStateFile(StateFileWriter& sw)
{
    /* write sections */
    for (objvec_type::iterator i = objects.begin(); i != objects.end(); ++i) {
        StateObjectWriter ow(sw, (*i)->getName(), (*i)->getFile()->getMtime());
        (*i)->writeStateFile(ow);
        ow.write();
    }
    
    /* write linker symbols */
    {
        StateObjectWriter ow(sw, "", 0);

        for (SymbolTable::iterator i = symbols.begin(); i != symbols.end(); ++i)
            if (i->second->object == 0)
                ow.addSymbol(i->first, *i->second);

        for (map_type::iterator i = sections.begin(); i != sections.end(); ++i)
            for (list_type::iterator j = i->second.begin(); j != i->second.end(); ++j)
                if ((*j)->getObject() == 0)
                    ow.addSection(**j);

//         ElfTableWriter<Elf32_Sym> sym_writer(sw.getWriter(), SHT_SYMTAB, state_sym_sec_name);
//         ElfStringWriter           sym_str_writer(sw.getWriter(), state_sym_str_name);
//         sw.getWriter().setLink(sym_writer.getIndex(), sym_str_writer.getIndex());
//         for (SymbolTable::iterator i = symbols.begin(); i != symbols.end(); ++i) {
//             if (i->second->object == 0) {
//                 Elf32_Sym s;
//                 s.st_name  = sym_str_writer.addString(i->first);
//                 s.st_value = i->second->address;
//                 if (i->second->section)
//                     s.st_value += i->second->section->getRelocationAddress();
//                 s.st_size  = 0;             // not used in ild
//                 s.st_info  = ELF32_ST_INFO(i->second->weak ? STB_WEAK : STB_GLOBAL, STT_NOTYPE);
//                 s.st_other = 0;
//                 s.st_shndx = SHN_ABS;
//                 sym_writer.add(s);
//             }
//         }
    }

    if (got_section)
        got_section->writeGotStateFile(sw);

    /* write common section */
    if (common_section) {
        ElfTableWriter<Elf32_Sym> sym_writer(sw.getWriter(), SHT_SYMTAB, state_common_sec_name);
        ElfStringWriter           sym_str_writer(sw.getWriter(), state_common_str_name);
        sw.getWriter().setLink(sym_writer.getIndex(), sym_str_writer.getIndex());
        for (common_type::iterator i = common_symbols.begin(); i != common_symbols.end(); ++i) {
            Elf32_Sym s;
            s.st_name  = sym_str_writer.addString(i->first);
            s.st_value = i->second.align /* actual value is in symbol section */;
            s.st_size  = i->second.size;
            s.st_info  = ELF32_ST_INFO(STB_GLOBAL, STT_NOTYPE);
            s.st_other = 0;
            s.st_shndx = SHN_ABS;
            sym_writer.add(s);
        }
    }
}
