/**
  *  \file vfile.cc
  *  \brief View on a File
  */
#include <algorithm>
#include <cstdio>
#include <iostream>
#include "vfile.h"
#include "archive.h"
#include "table.h"

/** \class VFile
    \brief View on a File

    This class provides a view on a file, or a "virtual sub-file".
    It offers read and write operations restricted to a particular
    subrange of positions in the file. This is used, for example, so
    that an archive member appears as a whole file.

    VFile sits atop a File; the File is shared between all VFiles
    pointing at it and closed when all VFiles are closed. */

/** Create view on a file. /afile/ is the real, physical file.
    To create a view from a VFile, use getSubFile(). */
VFile::VFile(Ptr<File> afile, off_t astart, off_t asize)
    : file(afile), start(astart), size(asize), mtime(0)
{
}

/** Read buffer. Returns number of bytes read. */
ssize_t VFile::read(off_t where, void* buffer, size_t count)
{
    if(size >= 0) {
        if(where > size)
            return 0;
        size_t max = size - where;
        if(max < count)
            count = max;
    }
    return file->read(where + start, buffer, count);
}

/** Write buffer. Returns number of bytes written. */
ssize_t VFile::write(off_t where, const void* buffer, size_t count)
{
    if(size >= 0) {
        if(where > size)
            return 0;
        size_t max = size - where;
        if(max < count)
            count = max;
    }
    return file->write(where + start, buffer, count);
}

/** Copy /count/ bytes in file /f/ from /from/ to /to/.
    Handles overlapping moves. */
void copyFileRegion(VFile& f, off_t from, off_t to, std::size_t count)
{
    if(from == to)
        return;
    
    const unsigned int MAX_SIZE = 2UL << 18;  // completely arbitrary
    if(count <= MAX_SIZE) {
        VLArray<char> buf(count);
        f.read(from, buf.getBuffer(), count);
        f.write(to, buf.getBuffer(), count);
        return;
    }

    /* file too large: copy blockwise */
    VLArray<char> buf(MAX_SIZE);
    if(from < to && from + count >= to) {
        /* overlapping: copy backwards */
        from += count;
        to += count;
        while(count != 0) {
            std::size_t now = count > MAX_SIZE ? MAX_SIZE : count;
            from -= now;
            to -= now;
            f.read(from, buf.getBuffer(), now);
            f.write(to, buf.getBuffer(), now);
            count -= now;
        }
    } else {
        /* non-overlapping */
        std::cerr << "Copying forward...\n";
        while(count != 0) {
            ssize_t now = count > MAX_SIZE ? MAX_SIZE : count;
            if(f.read(from, buf.getBuffer(), now) != now)
                std::perror("read");
            f.write(to, buf.getBuffer(), now);
            count -= now;
            from += now;
            to += now;
        }
    }
}

/** Create a view on a subrange of this file. */
Ptr<VFile>
VFile::getSubFile(off_t awhere, size_t acount)
{
    // FIXME: range checks &c
    return new VFile(file, start + awhere, acount);
}
