#ifndef ILD_TABLE_H
#define ILD_TABLE_H

#include <cstddef>
#include "cpluslib/types.h"

// type safe wrapper for tables of elements with variable
// (but fixed per table) size
template<class T>
class Table {
    char* buffer;
    std::size_t itemsize;
    std::size_t count;
 public:
    Table(std::size_t asize, std::size_t acount);
    ~Table();

    T& operator[](std::size_t n) {
        return *(T*)(buffer + n * itemsize);
    }
    const T& operator[](std::size_t n) const {
        return *(T*)(buffer + n * itemsize);
    }

    std::size_t getSize() const { return itemsize * count; }
    std::size_t getCount() const { return count; }
    char* getBuffer() { return buffer; }
    const char* getBuffer() const { return buffer; }
};

template<class T>
Table<T>::Table(std::size_t asize, std::size_t acount)
    : buffer(new char[asize * acount]),
      itemsize(asize),
      count(acount)
{ }

template<class T>
Table<T>::~Table()
{
    delete[] buffer;
}

// Variable length array of T's
template<class T>
class VLArray {
    VLArray(const VLArray&);
    VLArray& operator=(const VLArray&);
    
    T* buffer;
    std::size_t size;
 public:
    VLArray(std::size_t n)
        : buffer(new T[n]), size(n) { }
    ~VLArray() { delete[] buffer; }

    T& operator[](std::size_t n) const {
        return buffer[n];
    }

    void changeSize(std::size_t n) {
        T* nbuf = new T[n];
        delete[] buffer;
        buffer = nbuf;
        size = n;
    }

    T* getBuffer() { return buffer; }
    const T* getBuffer() const { return buffer; }
    std::size_t getSize() const { return size; }
};

template<class T>
uint32 getLongB(const T& t, std::size_t n) {
    return uint8(t[n]) * 16777216UL
        + uint8(t[n+1]) * 65536UL
        + uint8(t[n+2]) * 256UL
        + uint8(t[n+3]);
}

template<class T>
void putLongB(const T& t, std::size_t n, uint32 val) {
    t[n]   = val >> 24;
    t[++n] = val >> 16;
    t[++n] = val >> 8;
    t[++n] = val;
}

#endif
