/**
  *  \file symbol.h
  */
#ifndef ILD2_SYMBOL_H
#define ILD2_SYMBOL_H

#include <map>
#include "cpluslib/smartptr.h"
#include "elf.h"
#include "section.h"
#include "cpluslib/string.h"
#include "config.h"

enum { NO_TABLE_ADDRESS = 1 };

class LinkObject;
class ElfWriter;

struct Symbol {
    bool weak;                   // true if weak (un)definition
    bool defined;                // true if defined symbol
    Ptr<ProgramSection> section; // section (0 if absolute symbol)
    Elf32_Addr address;          // address (relative to section)
    bool iter_id;                // used for iteration through undefined symbols
    LinkObject* object;          // object where this symbol is defined

    Elf32_Addr got_offset;       // GOT offset, or NO_TABLE_ADDRESS if not known

    Symbol(bool aweak, bool adef, Ptr<ProgramSection> sec, Elf32_Addr adr, bool aiter_id,
           LinkObject* aobj);
    void define(bool aweak, bool adef, Ptr<ProgramSection> sec, Elf32_Addr adr, LinkObject* aobj);
};

class SymbolTable {
    typedef std::map<string_t, Ptr<Symbol> > symbol_map;

    symbol_map symbols;
    bool had_errors;
    bool iter_id;
    symbol_map::iterator iter;

 public:
    typedef symbol_map::const_iterator iterator;
    
    SymbolTable();

    void addSymbol(string_t name, bool weak, bool defined,
                   Ptr<ProgramSection> ref_sec, Elf32_Addr address,
                   LinkObject* obj);
    void addSymbol(string_t name, Ptr<Symbol> sym);
    Ptr<Symbol> getSymbol(string_t name) { return symbols[name]; }
#ifdef DEBUG
    void dump(std::ostream&);
#endif

    bool checkUndefinedSymbols();

    bool getFirstUndefinedSymbol(string_t& ret);
    bool getNextUndefinedSymbol(string_t& ret);
    void finishSymbolIteration();

    Elf32_Addr getAddressOf(string_t name);

    iterator begin() const { return symbols.begin(); }
    iterator end() const { return symbols.end(); }

    void clear();

    Elf32_Word saveRelocated(ElfWriter& w, string_t name);
};

extern SymbolTable symbols;

#endif
