/**
  *  \file symbol.cc
  */

#include "symbol.h"
#include "object.h"
#include "except.h"
#include "cpluslib/assert.h"
#include "log.h"
#include "config.h"
#include "elfio.h"

SymbolTable symbols;

//
//  [[Symbol]] - A Symbol (Un)Definition
//
//  Represents a (possibly undefined) symbol
//
Symbol::Symbol(bool aweak, bool adef, Ptr<ProgramSection> sec, Elf32_Addr adr,
               bool aiter_id, LinkObject* aobj)
    : weak(aweak),
      defined(adef),
      section(sec),
      address(adr),
      iter_id(aiter_id),
      object(aobj),
      got_offset(NO_TABLE_ADDRESS)
{ }

void
Symbol::define(bool aweak, bool adef, Ptr<ProgramSection> sec, Elf32_Addr adr,
               LinkObject* aobj)
{
    weak    = aweak;
    defined = adef;
    section = sec;
    address = adr;
    object  = aobj;
}

SymbolTable::SymbolTable()
    : had_errors(0), iter_id(false)
{
}

/** Define a symbol, from an already-filled in symbol.
    This is used when linking incrementally: the IncrementalObject
    collects many /Symbol/ objects and dumps them into the symbol
    table at once. You could also call addSymbol with the
    symbol specs taken apart, but this hints the symbol table to
    re-use an existing symbol object. */
void
SymbolTable::addSymbol(string_t name, Ptr<Symbol> sym)
{
    symbol_map::iterator i = symbols.find(name);
    if(i != symbols.end()) {
        /* symbol already defined. This can happen when an incrementally-
           linked module provides a (weak?) symbol already provided by
           a previous newly-linked symbol */
        addSymbol(name, sym->weak, sym->defined,
                  sym->section, sym->address, sym->object);
    } else {
        sym->iter_id = iter_id;
        symbols[name] = sym;
    }
}

void
SymbolTable::addSymbol(string_t name, bool weak, bool defined,
                       Ptr<ProgramSection> ref_sec, Elf32_Addr address,
                       LinkObject* obj)
{
    ASSERT(!ref_sec.ptr() || obj == ref_sec->getObject());
    symbol_map::iterator i = symbols.find(name);
    if(i == symbols.end()) {
        /* new symbol: add it */
        symbols[name] = new Symbol(weak, defined, ref_sec, address, iter_id, obj);
        return;
    }
    
    /* the symbol is already known */
    if(!defined) {
        /* undefined symbol. If this is a `strongly undefined' symbol,
           make the result strongly undefined, so that an error is
           generated when there's no definition */
        if(!i->second->defined)
            i->second->weak &= weak;
        return;
    }

    /* this is a symbol definition */
    if(i->second->defined) {
        /* symbol already defined */
        if(i->second->weak) {
            if(!weak)
                /* strong definition supersedes weak one */
                i->second->define(weak, defined, ref_sec, address, obj);
            else
                /* weak definition may occur any time -- FIXME: valid? */
                ;
        } else if(!weak) {
            /* two strong definitions */
            if(ref_sec.ptr() == i->second->section.ptr()
               && address == i->second->address) {
                /* *identical* redefinition, e.g., common symbol defined twice */
                return;
            }
            had_errors = true;
            log(LOG_ERROR) << "symbol conflict: " << name << " @ "
                           << Section::getSectionName(ref_sec)
                           << "\tvs. "
                           << Section::getSectionName(i->second->section) << std::endl;
        }
    } else {
        /* symbol not yet defined */
        i->second->define(weak, defined, ref_sec, address, obj);
    }
}

#ifdef DEBUG
void
SymbolTable::dump(std::ostream& os)
{
    static const char states[2][2] = { { 'U', 'D' }, { 'u', 'W' } };
    
    for(symbol_map::const_iterator i = symbols.begin(); i != symbols.end(); ++i) {
        Elf32_Addr a = 0;
        if(i->second->section.ptr()) {
            if(!i->second->section->isInImage())
                os << "ERROR: " << i->first << " defined relative to not-linked section\n";
            else
                a = i->second->section->getRelocationAddress();
        }
        
        os << states[i->second->weak][i->second->defined]
           << " "
           << Section::getSectionName(i->second->section)
           << " + " << i->second->address
           << "\t" << i->first << std::endl;
    }
}
#endif

bool
SymbolTable::checkUndefinedSymbols()
{
    bool errors = false;
    static bool unalloc_explain = false;
    for(symbol_map::const_iterator i = symbols.begin(); i != symbols.end(); ++i) {
        if(!i->second->defined) {
            if(i->second->weak) {
                /* undefined weak symbols have value zero */
                i->second->define(true, true, 0, 0, 0);
                log(LOG_INF) << "Undefined symbol `" << i->first << "' set to zero\n";
            } else {
                log(LOG_ERROR) << "Undefined symbol: " << i->first << std::endl;
                errors = true;
            }
        } else if(i->second->section.ptr()) {
            if(!i->second->section->isInImage()) {
                log(LOG_ERROR) << "Symbol `" << i->first << "' defined relative to unallocated section `"
                               << Section::getSectionName(i->second->section) << "'\n";
                if(!unalloc_explain && i->second->section->isLinkOnce()) {
                    log(LOG_ERROR) << "  This error can happen when the compiler generates inconsistent code\n";
                    log(LOG_ERROR) << "  in `linkonce' sections. Usually this means you have not recompiled all\n";
                    log(LOG_ERROR) << "  dependant files after changing a template definition.\n";
                    unalloc_explain = true;
                }
            }
        }
    }
    return errors;
}

//
//  Iteration through undefined symbols. Normally, all
//  /Symbol::iter_id/ fields have the same value as
//  /iter_id/. Iterating through the symbol map, we mark all processed
//  symbols by flipping that bit. This way, we can tell newly-added
//  symbols by simply looking at /iter_id/. At the end, all /iter_id/s
//  will have been flipped so we have to flip the one in this
//  /SymbolTable/ as well.
//
//  Implications of this implementation: there can be at most one
//  iteration active at a time. Iteration must be performed until the
//  end.
//
bool
SymbolTable::getFirstUndefinedSymbol(string_t& ret)
{
#ifdef DEBUG
    for(symbol_map::iterator i = symbols.begin(); i != symbols.end(); ++i)
        ASSERT(i->second->iter_id == iter_id);
#endif

    for(symbol_map::iterator i = symbols.begin(); i != symbols.end(); ++i) {
        /* first pass: no need to check iter_id */
        i->second->iter_id = !iter_id;
        if(!i->second->defined && !i->second->weak) {
            iter = i;
            ret = i->first;
            return true;
        }
    }
    iter_id = !iter_id;
    return false;
}

bool
SymbolTable::getNextUndefinedSymbol(string_t& ret)
{
    bool restarted = false;

    while(1) {
        if(++iter == symbols.end()) {
            if(!restarted) {
                iter = symbols.begin();
                restarted = true;
                ASSERT(iter != symbols.end()); // can't happen
            } else {
                iter_id = !iter_id;
                return false;
            }
        }
        if(iter->second->iter_id == iter_id) {
            iter->second->iter_id = !iter_id;
            if(!iter->second->defined && !iter->second->weak) {
                ret = iter->first;
                return true;
            }
        }
    }
}

void
SymbolTable::finishSymbolIteration()
{
    string_t s;
    while(getNextUndefinedSymbol(s))
        ;
}

Elf32_Addr
SymbolTable::getAddressOf(string_t name)
{
    symbol_map::iterator i = symbols.find(name);
    if(i == symbols.end())
        throw LinkerError("reference to undefined symbol `" + name + "'");

    ASSERT(i->second->defined);
    if(i->second->section.ptr())
        return i->second->section->getRelocationAddress() + i->second->address;
    else
        return i->second->address;
}

void
SymbolTable::clear()
{
    symbols.erase(symbols.begin(), symbols.end());
    ASSERT(symbols.empty());
    had_errors = iter_id = false;
}

Elf32_Word
SymbolTable::saveRelocated(ElfWriter& w, string_t name)
{
    /* is there a better way to do this? */
    Elf32_Word count = 0, length = 0;
    for (iterator i = begin(); i != end(); ++i)
        if (i->second->defined)
            ++count, length += i->first.length()+1;

    /* reserve memory */
    string_t            all_names;
    VLArray<Elf32_Sym>  symbols(count);
    all_names.reserve(length);

    /* fill it */
    Elf32_Word index = 0;
    for (iterator i = begin(); i != end(); ++i) {
        if (i->second->defined) {
            Elf32_Sym& s = symbols[index++];
            
            s.st_name  = all_names.length();
            s.st_value = i->second->address;
            if (i->second->section)
                s.st_value += i->second->section->getRelocationAddress();
            s.st_size  = 0;
            s.st_info  = ELF32_ST_INFO(i->second->weak ? STB_WEAK : STB_GLOBAL,
                                       STT_NOTYPE /* FIXME? */);
            s.st_other = 0;
            s.st_shndx = SHN_ABS;
            all_names.append(i->first);
            all_names.append(1, char(0));
        }
    }

    /* write it out */
    Elf32_Word sym_sec = w.addSection(SHT_SYMTAB, name);
    Elf32_Word str_sec = w.addSection(SHT_STRTAB, ".str" + name);
    w.getSection(sym_sec).sh_link    = str_sec;
    w.getSection(sym_sec).sh_entsize = sizeof(Elf32_Sym);
    w.setSectionData(sym_sec, symbols.getBuffer(), sizeof(Elf32_Sym) * count);
    w.setSectionData(str_sec, all_names.data(), all_names.length());
}
