/**
  *  \file state.h
  */
#ifndef ILD2_STATE_H
#define ILD2_STATE_H

#include "cpluslib/assert.h"
#include "cpluslib/buffer.h"
#include "elf.h"
#include "elfio.h"
#include "symbol.h"
#include "vfile.h"
#include "section.h"
#include "table.h"
#include "object.h"

class Section;

extern const char state_file_sec_name[],
    state_sym_prefix[], state_sec_prefix[],
    state_got_sec_name[], state_got_str_name[],
    state_sym_sec_name[], state_sym_str_name[],
    state_common_sec_name[], state_common_str_name[];

class StateFileWriter {
    ElfWriter                           writer;
    ElfTableWriter<Elf32_ild_FileEntry> file_writer;
    ElfStringWriter                     string_writer;
    bool written;
 public:
    StateFileWriter(Ptr<VFile> output);
    ~StateFileWriter();

    void addFile(const string_t& name, time_t time, Elf32_Word syms, Elf32_Word secs);
    void write();

    ElfWriter& getWriter() { return writer; }
};

class StateObjectWriter {
    StateFileWriter& w;
    ElfTableWriter<Elf32_Shdr>  section_writer;
    ElfTableWriter<Elf32_Sym>   sym_writer;
    ElfStringWriter             sec_str_writer, sym_str_writer;
    bool written;
 public:
    StateObjectWriter(StateFileWriter& w, const string_t& file, time_t time);
    ~StateObjectWriter();

    void addSection(const ProgramSection& sec);
    void addSymbol(const string_t& name, const Symbol& sym);
    void addUndefSymbol(const string_t& name, const Elf32_Sym& sym);

    void write();
};

class FileListSection : public Section {
    Table<Elf32_ild_FileEntry> entries;
    Ptr<StringTableSection>    strings;
 public:
    FileListSection(LinkObject* o, const Elf32_Shdr& hdr);
    Elf32_ild_FileEntry* getFileByName(const string_t& name);
    Elf32_ild_FileEntry& getFile(Elf32_Word w) { return entries[w]; }
    Elf32_Word           getFileIndexByName(const string_t& name);
    void afterCreate();
};

class SectionListSection : public Section {
    Table<Elf32_Shdr>        sections;
    Ptr<StringTableSection>  strings;
 public:
    SectionListSection(LinkObject* o, const Elf32_Shdr& hdr);
    void afterCreate();

    Elf32_Word  getCount() const { return sections.getCount(); }
    Elf32_Shdr& getSection(Elf32_Word index) { return sections[index]; }
    string_t    getNameOf(Elf32_Word index) { return strings->getString(sections[index].sh_name); }
};

class StateReader : private ElfObject {
    static Ptr<Section> createSection(LinkObject* o, const Elf32_Shdr& s);
    Ptr<FileListSection> files;
 public:
    StateReader(Ptr<VFile> file, string_t name);
    ~StateReader();

    Elf32_ild_FileEntry* getFileByName(const string_t& name) { return files->getFileByName(name); }
    Elf32_Word           getFileIndexByName(const string_t& name) { return files->getFileIndexByName(name); }
    Elf32_ild_FileEntry& getFile(Elf32_Word w) { return files->getFile(w); }

    /* don't call this one */
    void getInformation(SymbolTable&, MemoryAllocator&);

    using ElfObject::getSection;
    using ElfObject::getSectionByName;
};

#endif
