/**
  *  \file state.cc
  */

#include "state.h"
#include "cpluslib/assert.h"
#include "except.h"

/* State file format:

   Section name: ".ild.files"
   Section type: SHT_ild_FILELIST
   sh_link: string table (file names, ".str.ild.files")
   Contents: Elf32_ild_FileEntry[]
      ife_symbols  -> Section name: ".sym.<file>"
                      Section type: SHT_SYMTAB
                      sh_link: string table (symbol names)
                      Contents: Elf32_Sym[], all symbols absolute
      ife_sections -> Section name: ".sec.<file>"
                      Section type: SHT_ild_SECTIONS
                      sh_link: string table (section names)
                      Contents: Elf32_Shdr[], absolute addresses

   Section name: ".ild.got", ".ild.plt", ".ild.common", ".ild.syms"
   Section type: SHT_SYMTAB
   sh_link: string table (symbol names)
   Contents: Elf32_Sym[]

   Advantages over text format: easy to handle, no new data structures
   needed, load on demand easily possible.

   Disadvantages: ELF state files are almost twice as big(!) as text
   state files used before (e.g., one symbol in ELF is 41 bytes +
   symbol name, in text file it was ~15 bytes + name on average).

   Possible optimisations: string tables could be optimized (suffix
   encoding). Maybe invent a new SHT_ild_STRTAB (prefix encoding),
   this should gain a bit for file names (usually, you have some 100
   modules starting with "/usr/lib/libc.a"). The section names string
   table could be shared.
*/

const char        state_file_sec_name[]   = ".ild.files";
static const char state_file_str_name[]   = ".str.ild.files";
const char        state_sym_prefix[]      = ".sym.";
const char        state_sec_prefix[]      = ".sec.";
static const char state_sym_str_prefix[]  = ".str.sym.";
static const char state_sec_str_prefix[]  = ".str.sec.";
const char        state_got_sec_name[]    = ".ild.got";
const char        state_got_str_name[]    = ".str.ild.got";
const char        state_sym_sec_name[]    = ".ild.sym";
const char        state_sym_str_name[]    = ".str.ild.sym";
const char        state_common_sec_name[] = ".ild.common";
const char        state_common_str_name[] = ".str.ild.common";

StateFileWriter::StateFileWriter(Ptr<VFile> output)
    : writer(output, ET_ild_STATE),
      file_writer(writer, SHT_ild_FILELIST, state_file_sec_name),
      string_writer(writer, state_file_str_name),
      written(false)
{
    writer.setLink(file_writer.getIndex(), string_writer.getIndex());
}

StateFileWriter::~StateFileWriter()
{
    write();
}

void
StateFileWriter::write()
{
    if (written)
        return;
    written = true;
    string_writer.write();
    file_writer.write();
    writer.close();
}

void
StateFileWriter::addFile(const string_t& name, time_t time, Elf32_Word syms, Elf32_Word secs)
{
    Elf32_ild_FileEntry fe;
    fe.ife_name     = string_writer.addString(name);
    fe.ife_mtime    = time;
    fe.ife_symbols  = syms;
    fe.ife_sections = secs;
    file_writer.add(fe);
}

StateObjectWriter::StateObjectWriter(StateFileWriter& w, const string_t& file, time_t time)
    : w(w),
      section_writer(w.getWriter(), SHT_ild_SECTIONS, state_sec_prefix + file),
      sym_writer    (w.getWriter(), SHT_SYMTAB, state_sym_prefix + file),
      sec_str_writer(w.getWriter(), state_sec_str_prefix + file),
      sym_str_writer(w.getWriter(), state_sym_str_prefix + file),
      written(false)
{
    w.getWriter().setLink(section_writer.getIndex(), sec_str_writer.getIndex());
    w.getWriter().setLink(sym_writer.getIndex(),     sym_str_writer.getIndex());
    w.addFile(file, time, sym_writer.getIndex(), section_writer.getIndex());
}

StateObjectWriter::~StateObjectWriter()
{
    write();
}

void
StateObjectWriter::addSection(const ProgramSection& sec)
{
    Elf32_Shdr s;
    s.sh_name      = sec_str_writer.addString(sec.getName());
    s.sh_type      = sec.getType();
    s.sh_flags     = sec.getFlags();
    s.sh_addr      = sec.getAddress();
    s.sh_offset    = sec.getRelocationAddress();
    s.sh_size      = sec.getSize();
    s.sh_link      = SHN_UNDEF;
    s.sh_info      = 0;
    s.sh_addralign = sec.getAlignment();
    s.sh_entsize   = 0;
    section_writer.add(s);    
}

void
StateObjectWriter::addSymbol(const string_t& name, const Symbol& sym)
{
    Elf32_Sym s;
    s.st_name  = sym_str_writer.addString(name);
    s.st_value = sym.address;
    if (sym.section)
        s.st_value += sym.section->getRelocationAddress();
    s.st_size  = 0;             // not used in ild
    s.st_info  = ELF32_ST_INFO(sym.weak ? STB_WEAK : STB_GLOBAL, STT_NOTYPE);
    s.st_other = 0;
    s.st_shndx = SHN_ABS;
    sym_writer.add(s);
}

void
StateObjectWriter::addUndefSymbol(const string_t& name, const Elf32_Sym& sym)
{
    Elf32_Sym s = sym;
    s.st_name = sym_str_writer.addString(name);

    Ptr<Symbol> sym_ptr = ::symbols.getSymbol(name);
    ASSERT(sym_ptr.ptr());
    s.st_value = sym_ptr->address;
    if (sym_ptr->section)
        s.st_value += sym_ptr->section->getRelocationAddress();
    ASSERT(s.st_shndx == SHN_UNDEF);
    sym_writer.add(s);
}

void
StateObjectWriter::write()
{
    if (written)
        return;
    written = true;
    section_writer.write();
    sym_writer.write();
    sec_str_writer.write();
    sym_str_writer.write();
}

/****************************** StateReader ******************************/

StateReader::StateReader(Ptr<VFile> afile, string_t aname)
    : ElfObject(afile, aname, createSection)
{
    if (getType() != ET_ild_STATE)
        throw LinkerError("`" + name + "' is not a state file");

    files = getSectionByName(state_file_sec_name).cast<FileListSection>();
    if (!files)
        throw LinkerError("file " + name + " has missing/wrong " + state_file_sec_name + " section");
}

StateReader::~StateReader()
{ }

Ptr<Section>
StateReader::createSection(LinkObject* o, const Elf32_Shdr& s)
{
    switch(s.sh_type) {
     case SHT_ild_SECTIONS:
        return new SectionListSection(o, s);
     case SHT_ild_FILELIST:
        return new FileListSection(o, s);
     default:
        return Section::create(o, s);
    }
}

void
StateReader::getInformation(SymbolTable&, MemoryAllocator&)
{
    ASSERT(0);
}

FileListSection::FileListSection(LinkObject* o, const Elf32_Shdr& s)
    : Section(o, s),
      entries(header.sh_entsize, header.sh_size / header.sh_entsize)
{
    o->getFile()->read(s.sh_offset, entries.getBuffer(), entries.getSize());
}

void
FileListSection::afterCreate()
{
    strings = checked_cast<ElfObject*>(getObject())->getSection(header.sh_link).cast<StringTableSection>();
    if (!strings)
        throw LinkerError("invalid sh_link in file list section `" + name + "'");
    strings->load();
}

Elf32_ild_FileEntry*
FileListSection::getFileByName(const string_t& name)
{
    for (Elf32_Word w = 0; w < entries.getCount(); ++w)
        if (name == strings->getString(entries[w].ife_name))
            return &entries[w];
    return 0;
}

Elf32_Word
FileListSection::getFileIndexByName(const string_t& name)
{
    for (Elf32_Word w = 0; w < entries.getCount(); ++w)
        if (name == strings->getString(entries[w].ife_name))
            return w;
    return Elf32_Word(-1);
}

SectionListSection::SectionListSection(LinkObject* o, const Elf32_Shdr& s)
    : Section(o, s),
      sections(header.sh_entsize, header.sh_size / header.sh_entsize)
{
    o->getFile()->read(s.sh_offset, sections.getBuffer(), sections.getSize());
}

void
SectionListSection::afterCreate()
{
    strings = checked_cast<ElfObject*>(getObject())->getSection(header.sh_link).cast<StringTableSection>();
    if (!strings)
        throw LinkerError("invalid sh_link in section list section `" + name + "'");
    strings->load();
}
