/**
  *  \file section1.cc
  *  \brief Parts of the `Section' module which are needed by `iar'.
  *
  *  Having these separate avoids linking in almost all of `ild'
  *  into `iar'. `iar' uses only ad-hoc symbol table parsing.
  */
#include "section.h"
#include "vfile.h"
#include "cpluslib/assert.h"
#include "except.h"

/** \class Section
    \brief ELF file section

    Objects of this class represents a single section of an
    ELF object file, or synthetic sections.

    Objects of this type (and derived classes) are normally created
    using Section::create; do not call the constructor directly. */

Section::Section(LinkObject* aobject, const Elf32_Shdr& aheader)
    : object(aobject), header(aheader), state_id(0)
{ }

Section::~Section()
{
}

/** Called when the object file has been completely loaded (that is,
    all /Section/ objects are known), but before linking proceeds. This
    can be used to set up pointers to ``peer'' sections if needed. */
void
Section::afterCreate()
{ }

/** Register symbols and sections with the specified symbol table and
    virtual memory allocator. This prepares everything needed to
    compute the linkee's virtual memory image.

    - symbol section: register symbols in symbol table
    - program sections: register section with memory allocator
    - relocation sections: these contains instructions regarding the GOT
      which must be told to the memory allocator
    
    See also LinkObject::getInformation. */
void
Section::getInformation(SymbolTable& /*symtab*/, MemoryAllocator& /*memalloc*/)
{ }

/** Write state file information for this section. */
void
Section::writeStateFile(StateObjectWriter&)
{ }

/****************************** String Table *****************************/

/** \class StringTableSection
    \brief String table (SHT_STRTAB)

    This class provides a convenient interface to access ELF string
    tables. It provides "explicit demand loading": before attempting
    to get a string from a StringTableSection object, you must have
    called load() or loadFrom(). Essentially, this means that when
    loading a normal ELF object, only the string table corresponding to
    section and symbol names will be loaded, not the debug information
    sections. */
StringTableSection::StringTableSection(LinkObject* o, const Elf32_Shdr& h)
    : Section(o, h), table(0), loaded(false)
{ }

/** Load from specified VFile. When this section has a valid object
    pointer (first parameter to ctor non-null), use load() instead
    which will automatically use the right VFile. */
void
StringTableSection::loadFrom(VFile& file)
{
    if(loaded)
        return;

    loaded = true;
    table.changeSize(header.sh_size);
    file.read(header.sh_offset, table.getBuffer(), header.sh_size);

    /* make sure that all strings are zero-terminated */
    while(header.sh_size && table[header.sh_size-1] != 0)
        --header.sh_size;
    if(!header.sh_size)
        throw LinkerError("invalid string table");
}

/** Get the string pointed to by \c offs. Throws an error if \c offs is
    out of range. \pre load() or loadFrom() has been called. */
string_t
StringTableSection::getString(Elf32_Word offs)
{
    ASSERT(loaded);
    if(offs >= header.sh_size)
        throw LinkerError("Invalid string table entry requested");

    return table.getBuffer() + offs;
}
