/**
  *  \file section.h
  *  \brief ELF sections
  *
  *  Parts of this are used by iar, too.
  */
#ifndef ILD_SECTION_H
#define ILD_SECTION_H

#include <string>
#include "elf.h"
#include "table.h"
#include "cpluslib/string.h"
#include "cpluslib/smartptr.h"

class LinkObject;
class ProgramSection;
class SymbolTable;
class MemoryAllocator;
class Output;
class RelocationSection;
class VFile;
class StateObjectWriter;

class Section {
 protected:
    LinkObject* object;         ///< backlink to object file, may be 0
    Elf32_Shdr  header;         ///< section header
    string_t    name;           ///< name (set by loader code)

    Section(LinkObject* aobject, const Elf32_Shdr& aheader);

    int state_id;
 public:
    virtual ~Section();
    virtual void afterCreate();

    /* Get symbolic information: section names, symbols */
    virtual void getInformation(SymbolTable& symtab, MemoryAllocator& memalloc);
    virtual void writeStateFile(StateObjectWriter&);

    /* accessors */
    void        setName(string_t aname) { name = aname; }
    string_t    getName() const         { return name; }
    LinkObject* getObject() const       { return object; }

    Elf32_Word getAlignment() const       { return header.sh_addralign; }
    Elf32_Word getSize() const            { return header.sh_size; }
    void       setAlignment(Elf32_Word a) { header.sh_addralign = a; }
    void       setSize(Elf32_Word s)      { header.sh_size = s; }
    Elf32_Word getType() const            { return header.sh_type; }
    Elf32_Word getFlags() const           { return header.sh_flags; }

    void setStateId(int sid) { state_id = sid; }
    int  getStateId() const  { return state_id; }

    /* utilities */
    static Ptr<Section> create(LinkObject* obj, const Elf32_Shdr& hdr);
    static string_t getSectionName(Ptr<Section> section);
};

/* String table */
class StringTableSection : public Section {
    VLArray<char> table;        ///< contents
    bool loaded;                ///< true iff section was loaded
 public:
    StringTableSection(LinkObject* o, const Elf32_Shdr& h);
    string_t getString(Elf32_Word offs);
    void load();
    void loadFrom(VFile& f);
};

/* Symbol Section */
class SymbolSection : public Section {
    Table<Elf32_Sym> symbols;              ///< all symbol table entries
    Ptr<StringTableSection> symbol_names;  ///< related string section (with names)

    inline void getSymbol(SymbolTable& symtab, MemoryAllocator& memalloc,
                          const Elf32_Sym& sym);
 public:
    SymbolSection(LinkObject* o, const Elf32_Shdr& h);

    void afterCreate();
    void getInformation(SymbolTable& symtab, MemoryAllocator& memalloc);
    void writeStateFile(StateObjectWriter&);
    Elf32_Word getAddressOf(Elf32_Word index);
    string_t getNameOf(Elf32_Word index);
};

class ProgramSection : public Section {
 protected:
    /** Address in virtual memory (load address). Section contents is
        loaded here. */
    Elf32_Addr address;

    /** Relocation address (for "phased" code). Relocations are
        computed relative to this one. */
    Elf32_Addr reloc_address;

    /** true for linkonce sections. Of many linkonce sections with the
        same name, we pick one at will and link only that. */
    bool linkonce;

    /** true if section is part of virtual memory image (i.e., it has
        an address). */
    bool in_image;

    /** true if section must be written out. */
    bool need_write;

    /** true if this section was visited by linker script
        (used to emit warnings) */
    bool visited;

    ProgramSection(LinkObject* o, const Elf32_Shdr& h);
 public:
    void getInformation(SymbolTable& symtab, MemoryAllocator& memalloc);
    void writeStateFile(StateObjectWriter&);

    /* Accessors */
    void setAddress(Elf32_Addr aadr);
    void setRelocationAddress(Elf32_Addr aradr);
    Elf32_Addr getAddress() const { return address; }
    Elf32_Addr getRelocationAddress() const { return reloc_address; }

    bool isLinkOnce() const { return linkonce; }
    bool isInImage() const { return in_image; }
    bool isVisited() const { return visited; }
    bool needsToBeWritten() const { return need_write; }

    void markVisited() { visited = true; }
    void markNotWrite() { need_write = false; }

    /* Write output to file */
    virtual void writeOutput(Output& out) = 0;
};

/* Program Section from ELF file */
class ProgbitsSection : public ProgramSection {
    RelocationSection* relsec;         ///< RelSection has a backlink
 public:
    ProgbitsSection(LinkObject* o, const Elf32_Shdr& h);

    void afterCreate();
    void writeOutput(Output& out);
    void setRelocation(RelocationSection* arelsec);
};

/* Section with relocation entries */
class RelocationSection : public Section {
 protected:
    RelocationSection(LinkObject* o, const Elf32_Shdr& h);
 public:
    virtual void relocate(unsigned char* data, std::size_t size) = 0;
};

/* Section with relocation entries, implicit addends */
class RelSection : public RelocationSection {
    Ptr<SymbolSection> sym_sec;      ///< related symbol table.
    Ptr<ProgbitsSection> prog_sec;   ///< related program section.
    Table<Elf32_Rel> rel;
  public:
    RelSection(LinkObject* o, const Elf32_Shdr& h);

    void afterCreate();

    void relocate(unsigned char* data, std::size_t size);
    void getInformation(SymbolTable& symtab, MemoryAllocator& memalloc);
};

#endif
