/**
  *  \file section.cc
  *  \brief ELF sections (most of them)
  *
  *  Almost every ELF input section is represented by a descendant
  *  of Section.
  */

#include "section.h"
#include "object.h"
#include "cpluslib/assert.h"
#include "except.h"
#include "symbol.h"
#include "vmalloc.h"
#include "table.h"
#include "namemap.h"
#include "output.h"
#include "log.h"
#include "names.inc"
#include "state.h"

/******************************** Section ********************************/

/** Create a section. Creates a section of appropriate (child) class.
    \param hdr    section header (from ELF section table)
    \param obj    object file */
Ptr<Section>
Section::create(LinkObject* obj, const Elf32_Shdr& hdr)
{
    switch(hdr.sh_type) {
     case SHT_STRTAB:
        return new StringTableSection(obj, hdr);
     case SHT_PROGBITS:
     case SHT_NOBITS:
        return new ProgbitsSection(obj, hdr);
     case SHT_SYMTAB:
        return new SymbolSection(obj, hdr);
     case SHT_REL:
        return new RelSection(obj, hdr);
     default:
        return new Section(obj, hdr);
    }
}

/** Get name of a section in human-readable format. Section may be
    null (used by absolute symbols) or a section not in any object. */
string_t
Section::getSectionName(Ptr<Section> section)
{
    if(!section.ptr())
        return "<absolute>";
    if(section->getObject())
        return section->getObject()->getName() + "::" + section->getName();
    else
        return section->getName();
}

/*************************** StringTableSection **************************/

/** Load string table. Call this method before a possible call to
    getString. \pre section belongs to an object */
void
StringTableSection::load()
{
    loadFrom(*object->getFile());
}

/***************************** SymbolSection *****************************/

/** \class SymbolSection
    \brief Symbol Table (SHT_SYMTAB, SHT_DYNSYM)

    This class manages symbol tables. We assume all symbols listed
    as undefined in the symbol table are actually needed. */

SymbolSection::SymbolSection(LinkObject* o, const Elf32_Shdr& h)
    : Section(o, h),
      symbols(h.sh_entsize, h.sh_size / h.sh_entsize)
{
    // FIXME: error checking?
    o->getFile()->read(h.sh_offset, symbols.getBuffer(), symbols.getSize());
}

/** After creation: fetch associated string table */
void
SymbolSection::afterCreate()
{
    symbol_names = checked_cast<ElfObject*>(object)->getSection(header.sh_link).cast<StringTableSection>();
    if(!symbol_names.ptr())
        throw LinkerError("symbol section refers to invalid string table");
    symbol_names->load();
}

/** Process a single symbol. Attempts to merge the specified symbol
    with the current knowledge of the memory allocator and the symbol
    table (i.e. symbol is (un)defined as necessary). */
inline void
SymbolSection::getSymbol(SymbolTable& symtab, MemoryAllocator& memalloc,
                         const Elf32_Sym& sym)
{
    string_t name = symbol_names->getString(sym.st_name);
    bool weak;

    /* determine symbol "weakness" */
    if(ELF32_ST_BIND(sym.st_info) == STB_WEAK)
        weak = true;
    else if(ELF32_ST_BIND(sym.st_info) == STB_GLOBAL)
        weak = false;
    else
        throw LinkerError("Invalid symbol binding for `" + name + "'");

    /* attempt to define the symbol */
    switch(sym.st_shndx) {
     case SHN_UNDEF:
        /* undefined symbol */
        symtab.addSymbol(name, weak, false, 0, 0, 0);
        break;
     case SHN_COMMON:
        /* common symbol */
        memalloc.addCommon(name, sym.st_size, sym.st_value);
        break;
     case SHN_ABS:
        /* absolute symbol */
        symtab.addSymbol(name, weak, true, 0, sym.st_value, object);
        break;
     default:
        /* symbol defined relative to a section. No problem for symbol
           tables from an object file (ElfObject), but for incremental
           linking (IncrementalObject). Normally, all objects in state
           files are defined absolute and this does not occur. However
           we don't want ild to crash when someone does something strange. */
        ElfObject* elf_o = dynamic_cast<ElfObject*>(object);
        if (!elf_o)
            throw LinkerError("this can't happen (invalid state file?)");
        Ptr<ProgramSection> ps =
            elf_o->getSection(sym.st_shndx).cast<ProgramSection>();
        if(!ps.ptr())
            throw LinkerError("Symbol `" + name + "' defined relative to invalid section");
        symtab.addSymbol(name, weak, true, ps, sym.st_value, object);
        break;
    }
}

/** Get symbolic information: process all symbols and put them into
    the symbol table / common section. */
void
SymbolSection::getInformation(SymbolTable& symtab, MemoryAllocator& memalloc)
{
    ASSERT(symbol_names.ptr());
    for(Elf32_Word i = header.sh_info; i < symbols.getCount(); ++i)
        getSymbol(symtab, memalloc, symbols[i]);
}

void
SymbolSection::writeStateFile(StateObjectWriter& w)
{
    for (Elf32_Word i = header.sh_info; i < symbols.getCount(); ++i) {
        Elf32_Sym& sym  = symbols[i];
        string_t   name = symbol_names->getString(sym.st_name);
        if (sym.st_shndx == SHN_UNDEF) {
            w.addUndefSymbol(name, sym);
        } else {
            Ptr<Symbol> sym_ptr = ::symbols.getSymbol(name);
            ASSERT(sym_ptr.ptr());
            if (sym_ptr->object == this->getObject())
                w.addSymbol(name, *sym_ptr);
        }
    }
}

/** Fetch address of a symbol.
    \param index index of symbol as stored in a relocation entry. */
Elf32_Word
SymbolSection::getAddressOf(Elf32_Word index)
{
    if(index < header.sh_info) {
        /* local symbol -- resolve manually */
        switch(symbols[index].st_shndx) {
         case SHN_UNDEF:
            throw LinkerError("reference to undefined local symbol");
         case SHN_COMMON:
            throw LinkerError("Impossible: local common: " + symbol_names->getString(symbols[index].st_name));
         case SHN_ABS:
            return symbols[index].st_value;
         default:
            Ptr<ProgramSection> ps =
                checked_cast<ElfObject*>(object)->getSection(symbols[index].st_shndx).cast<ProgramSection>();
            if(!ps.ptr())
                throw LinkerError("symbol defined relative to non-program section");
            if (!ps->isInImage())
                /* symbol defined relative to a non-linked section:
                   this *can* happen: when a template function needs
                   exception frames, these frames (not linkonce) will
                   contain references to the function incarnation in
                   this object (linkonce text). */
                return 0;
            return ps->getRelocationAddress() + symbols[index].st_value;
        }
    } else {
        /* global symbol */
        return ::symbols.getAddressOf(symbol_names->getString(symbols[index].st_name));
    }
}

/** Return name of symbol /index/. This is needed for creating GOT
    entries to local symbols. */
string_t
SymbolSection::getNameOf(Elf32_Word index)
{
    if(index < header.sh_info) {
        /* We can't just return the actual symbol name for local symbols.
           Instead, prepend the name of the object file and a colon:
           this will make the name as unique as needed, but makes symbol
           dumps still user-readable. */
        ASSERT(object);
        return object->getName() + ":" + symbol_names->getString(symbols[index].st_name);
    } else {
        return symbol_names->getString(symbols[index].st_name);
    }
}

/***************************** ProgramSection ****************************/

/** \class ProgramSection
    \brief Section that contributes to VM image

    Represents a section of the linkee's virtual memory image.
    Subclasses represent actual source sections or synthesized
    ones. */
ProgramSection::ProgramSection(LinkObject* o, const Elf32_Shdr& h)
    : Section(o, h), address(0), reloc_address(0), linkonce(false),
      in_image(false), need_write(false), visited(false)
{ }

/** Get information: registers the section at the memory allocator. */
void
ProgramSection::getInformation(SymbolTable& /*symtab*/, MemoryAllocator& memalloc)
{
    if(header.sh_flags & SHF_ALLOC)
        memalloc.addSection(this);
}

void
ProgramSection::writeStateFile(StateObjectWriter& w)
{
    if (isInImage())
        w.addSection(*this);
}

/** Set the address of this section. Called by the memory
    allocator. The section will not be linked when this has not been
    called. This also sets the relocation address to be the same as
    the memory address (load address). */
void
ProgramSection::setAddress(Elf32_Addr aadr)
{
    need_write = in_image = true;
    reloc_address = address = aadr;
}

/** Set relocation address. When the relocation address differs from
    the load address, the section must be memcpy'd to the relocation
    address before it can be used. */
void
ProgramSection::setRelocationAddress(Elf32_Addr aradr)
{
    reloc_address = aradr;
}

/**************************** ProgbitsSection ****************************/

/** \class ProgbitsSection
    \brief Program section from input file (SHT_NOBITS, SHT_PROGBITS)

    A program section, usually from input file. */

ProgbitsSection::ProgbitsSection(LinkObject* o, const Elf32_Shdr& h)
    : ProgramSection(o, h), relsec(0)
{ }

void
ProgbitsSection::afterCreate()
{
    linkonce = (name.compare(0, 14, ".gnu.linkonce.", 14) == 0);
}

void
ProgbitsSection::writeOutput(Output& out)
{
    if(!needsToBeWritten())
        return;

    int prot = PF_R;
    if(header.sh_flags & SHF_WRITE)
        prot |= PF_W;
    if(header.sh_flags & SHF_EXECINSTR)
        prot |= PF_X;

    if(header.sh_type == SHT_PROGBITS) {
        ASSERT(object);
        VLArray<unsigned char> data(header.sh_size);
        object->getFile()->read(header.sh_offset, data.getBuffer(), header.sh_size);
        if(relsec)
            relsec->relocate(data.getBuffer(), data.getSize());
        out.writeOutput(data.getBuffer(), address, header.sh_size, header.sh_size, prot);
    } else {
        ASSERT(header.sh_type == SHT_NOBITS);
        out.writeOutput(0, address, 0, header.sh_size, prot);
    }
}

void
ProgbitsSection::setRelocation(RelocationSection* arelsec)
{
    if(relsec)
        throw LinkerError(getSectionName(this), "multiple relocation sections");
    relsec = arelsec;
}

/*************************** RelocationSection ***************************/

/** \class RelocationSection
    \brief Relocations

    A section containing relocations. This is a generic interface to
    support multiple relocation types. */

RelocationSection::RelocationSection(LinkObject* o, const Elf32_Shdr& h)
    : Section(o, h)
{ }

/******************************* RelSection ******************************/

/** \class RelSection
    \brief Relocation section with implicit addends (SHT_REL)

    With this type, addends are implicit (stored in target section
    locations that are to be relocated). This is the type usually
    used on Linux. */

RelSection::RelSection(LinkObject* o, const Elf32_Shdr& h)
    : RelocationSection(o, h),
      rel(header.sh_entsize, header.sh_size / header.sh_entsize)
{
    o->getFile()->read(header.sh_offset, rel.getBuffer(), rel.getSize());
}

/** After creation: figure out related sections (symbols, program) */
void
RelSection::afterCreate()
{
    prog_sec = checked_cast<ElfObject*>(object)->getSection(header.sh_info).cast<ProgbitsSection>();
    if(!prog_sec.ptr())
        throw LinkerError("invalid sh_info in section `" + name + "'");
    prog_sec->setRelocation(this);

    sym_sec = checked_cast<ElfObject*>(object)->getSection(header.sh_link).cast<SymbolSection>();
    if(!sym_sec.ptr())
        throw LinkerError("invalid sh_link in section `" + name + "'");
}

/** Relocate. data points to target data. This will edit the data
    for the relocation (note that this is not idempotent). */
void
RelSection::relocate(unsigned char* data, std::size_t size)
{
    Elf32_Word got_addr = 0;
    bool warn = false;
    for(std::size_t i = 0; i < rel.getCount(); ++i) {
        Elf32_Word offset = rel[i].r_offset;
        int sym           = ELF32_R_SYM(rel[i].r_info);
        int type          = ELF32_R_TYPE(rel[i].r_info);
        Elf32_Word addend = *(Elf32_Word*)(data+offset);

        if (size < 4 || offset > size-4) {
            if (!warn)
                log(LOG_ERROR) << "relocation offset out of range in `" << getSectionName(this) << "'\n";
            warn = true;
            continue;
        }

        // FIXME -- handle misalignment

        switch(type) {
         case R_386_32:
         case R_386_PC32:
            /* FIXME: lots of optimisation needed */
            {
                int value = sym_sec->getAddressOf(sym) + addend;
                if(type == R_386_PC32)
                    value -= prog_sec->getRelocationAddress() + offset;
                *(Elf32_Word*)(data+offset) = value;
                break;
            }
         case R_386_PLT32:
            /* Offset from PC to PLT slot for symbol, used in
               call/jmpl insns. When statically linking, we
               behave as R_386_PC32 (FIXME: is this allowed?) */
            {
                int value = sym_sec->getAddressOf(sym) + addend
                    - prog_sec->getAddress() - offset;
                *(Elf32_Word*)(data+offset) = value;
                break;
            }
         case R_386_GOTOFF:
            /* S + A - GOT: symbol relative to GOT, but not addressed
               through GOT(!). Used to address static data of a library
               without using a GOT entry for it. */
            {
                if(!got_addr)
                    got_addr = symbols.getAddressOf(got_sym_name);
                *(Elf32_Word*)(data+offset) =
                    sym_sec->getAddressOf(sym) + addend - got_addr;
                break;
            }
         case R_386_GOTPC:
            /* GOT + A - PC: offset from PC to GOT. Used to get the
               GOT address from PIC. */
            {
                if(!got_addr)
                    got_addr = symbols.getAddressOf(got_sym_name);
                *(Elf32_Word*)(data+offset) =
                    got_addr + addend - prog_sec->getAddress() - offset;
                break;
            }
         case R_386_GOT32:
            /* G + A - GOT, where G = address of GOT entry of symbol.
               Used to get address of a symbol when we have the GOT
               address. */
            {
                *(Elf32_Word*)(data+offset) =
                    addend + symbols.getSymbol(sym_sec->getNameOf(sym))->got_offset;
                break;
            }
         default:
//             throw LinkerError("Unsupported relocation type "
//                               + ::getName(r_names, type));
            log(LOG_ERROR) << "Unsupported relocation type "
                           << ::getName(r_names, type) << "\n";
        }
    }
}

/** Gather information: create GOT entries as requested by this
    section. */
void
RelSection::getInformation(SymbolTable& /*symtab*/, MemoryAllocator& memalloc)
{
    for(std::size_t i = 0; i < rel.getCount(); ++i) {
        int sym  = ELF32_R_SYM(rel[i].r_info);
        int type = ELF32_R_TYPE(rel[i].r_info);
        switch(type) {
         case R_386_GOTOFF:
            /* S + A - GOT: symbol relative to GOT, but not addressed
               through GOT(!) */
            memalloc.createGot();
            break;
         case R_386_GOTPC:
            /* GOT + A - PC: offset from PC to GOT. */
            memalloc.createGot();
            break;
         case R_386_GOT32:
            /* G + A - GOT, where G = offset of GOT entry of symbol */
            memalloc.getGotOffset(sym_sec->getNameOf(sym));
            break;
        }
    }
}
