#!/usr/bin/perl -w
#
#  FAST dependency generator
#
#  `gcc -E -MM' is slow (because it does FULL preprocessing, starting
#  over at each single file), bogus (gets names of .o files in subdirs
#  wrong, at least in gcc-2.x) and too precise (I don't want
#  dependencies on my system's library in a public makefile). On the
#  plus side is that it is definitely correct in parsing gcc
#  options and #defines...
#
#  This one is mindblazingly fast (because each file is read only
#  once: about 10 times faster than `gcc') makes generic dependencies
#  (no system includes, but #ifdef-out'ed ones [for compile-time
#  switches]), and portable (doesn't need gcc).
#
#  (c) copyright 2002-2004 Stefan Reuther <Streu@gmx.de>
#

use strict;

my $usage = "$0 - FAST dependency generator

Usage: $0 [-I<dir>] [-e<ext>] [\@<file>] [-autoscan=<dir>] [<files>...]

  -I<dir>          add dir to include search path;
  -e<ext>          set extension for object files (e.g. `-eo' sets the
                   default of `.o');
  \@<file>          <file> contains a list of source files, one per line
                   (<file> can be `-' to read list from stdin). Only lines
                   ending in .c .cc .cpp .cxx .C .c++ will be accepted
                   (only those files that can be C/C++-compiled), blank
                   lines and lines starting with `;' are ignored;
  <files>...       files to process. This list is not filtered like the
                   @<file> list;
  -autoscan=<dir>  automatically scan for source files, recursively,
                   starting at <dir>.
  -makefile[=file] read makefile for generated file names (i.e. include
                   them in dependencies if needed, don't claim they are
                   missing if they don't exist)

Except for -e, all options are cumulative.

For plug-compatibility with gcc (CFLAGS), options starting
with -D -L -l -g -O -W -f -E -M -ansi -pedantic -pipe are ignored.

\(c) 2002-2004 Stefan Reuther <Streu\@gmx.de>
";

my @IPATH = (".");
my @FILES = ();
my $EXT = "o";
my $SOURCE = '\.(c|cc|cpp|cxx|C|c\+\+)';
my @autoscan = ();
my %autoscan_helper = ();
my %existing_files = ();        # files declared to exist

## Parameter Parsing

foreach(@ARGV) {
    if (/^--?help$/ || /^-h$/) {
	print $usage;
	exit 0;
    } elsif (/^-I(.+)/) {
	push @IPATH, $1;
    } elsif (/^-e(.+)/) {
	$EXT = $1;
    } elsif (/^-autoscan=(.*)/) {
	push @autoscan, $1;
    } elsif (/^-makefile=(.*)/) {
        &read_makefile ($1);
    } elsif (/^-makefile$/) {
        if (-f "Makefile") {
            &read_makefile ("Makefile");
        } elsif (-f "makefile") {
            &read_makefile ("makefile");
        } else {
            print STDERR "Unable to find makefile\n";
            exit 1;
        }
    } elsif (/^-[DLlgOWfEM]/ || /^-(ansi|pedantic|pipe)/) {
	# pp define, lib directory, lib, debug, optimize, warning, flag
	# --> ignore these
    } elsif ($_ eq '@-') {
	&add_files_from_fd (\*STDIN);
    } elsif (/^@(.*)/) {
	open FILE, "< $1" or do {
	    print STDERR "unable to open list file `$1': $!";
	    exit 1;
	};
	&add_files_from_fd (\*FILE);
	close FILE;
    } elsif (/^-/) {
	print STDERR "unknown option `$_' has been ignored\n";
    } else {
	push @FILES, $_;
    }
}

## Autoscan for source (recursive)

if (@autoscan) {
    # be careful not to include a file twice
    foreach (@FILES) {
	$autoscan_helper{$_} = 1;
    }
    # breadth-first search (uses less resources than depth-first)
    while (@autoscan) {
	my $now = shift @autoscan;
	if ($autoscan_helper{$now}) {
	    # already been here
	    next;
	}
	$autoscan_helper{$now} = 1;
	opendir DIR, "$now" or do {
	    print STDERR "$now: $!\n";
	    next;
	};
	while (defined (my $f = readdir DIR)) {
	    my $name = "$now/$f";
	    $name =~ s,^\./,,;
	    next if -l $name;
	    if (-d $name) {
		next if $f eq "." || $f eq "..";
		push @autoscan, $name;
	    } elsif (-f $name && $f =~ /$SOURCE$/ && !defined $autoscan_helper{$name}) {
		$autoscan_helper{$name} = 1;
		push @FILES, $name;
	    }
	}
	closedir DIR;
    }
    %autoscan_helper = ();	# free memory
}

## Scan Input

my @workfiles = @FILES;
my %realfile = ();		# maps logical file names to physical
my %done = ();
my %depend = ();

while (@workfiles) {
    my $now = shift @workfiles;
    open FILE, "< $now" or do {
	print STDERR "$now: $!\n";
	next;
    };
    %{$depend{$now}} = ();
    while (<FILE>) {
	if (/^\s*\#\s*include\s*\"([^\"]+)\"/) {
	    my $fn = &find_file($1, $now);
	    if (defined $fn) {
		$depend{$now}{$fn} = 1;
		if (!defined $done{$fn}) {
		    push @workfiles, $fn;
		    $done{$fn} = 1;
		}
	    }
	}
    }
    close FILE;
}

## Generate Output

$EXT = ".$EXT" unless $EXT =~ /^\./;
foreach (@FILES) {
    # all files on which $_ depends (directly or indirectly) are
    # in %{$depend{$_}}
    my %out = ($_ => 1);		# here, we collect dependencies
    my @work = keys %{$depend{$_}};
    while (@work) {
	my $now = shift @work;
	$out{$now} = 1;
	foreach (keys %{$depend{$now}}) {
	    push @work, $_
		unless defined $out{$_};
	}
    }
    # Okay, now generate output
    my $goal = $_; $goal =~ s/\.[^.]+$/$EXT/;
    my $line = "$goal:";
    my $done = 0;
    foreach (sort keys %out) {	# order is unimportant, but it looks better
	my $fn = $_;
	if (length($fn) + length($line) > 70) {
	    print "$line \\\n";
	    $line = "";
	}
	$line .= " $fn";
	$done = 1;
    }
    print "$line\n" if $done;	# don't show empty rules
}
exit 0;

sub find_file {
    my $fn = shift;
    my $inc = shift;
    if (exists $realfile{$fn}) {
	return $realfile{$fn};
    }
    foreach (@IPATH) {
	my $thisfn = "$_/$fn";
	$thisfn =~ s,^\./,,;
	if (-r "$thisfn") {
	    $realfile{$fn} = "$thisfn";
	    return $thisfn;
	}
        if (exists $existing_files{$thisfn}) {
            print STDERR "pretending that $thisfn exists\n";
            $done{$fn} = 1;             # so that worker code won't attempt to open it
            $realfile{$fn} = "$thisfn";
            return $thisfn;
        }
    }
    print STDERR "$inc: unable to find `$fn'\n";
    $realfile{$fn} = undef;
    return undef;
}

sub add_files_from_fd {
    my $fd = shift;
    while (<$fd>) {
	chomp;
	next if /^\s*$/ || /^\s*;/;
	push @FILES, $_
	    if /$SOURCE$/;
    }
}

sub read_makefile {
    my $fn = shift;
    open MF, "< $fn" or do {
        print STDERR "can't open $fn: $!\n";
        exit 1;
    };
    while (<MF>) {
        if (/^([^=]*\S)\s*:/) {
            foreach (split /\s+/, $1) {
                $existing_files{$_} = 1
                    unless /^\./;
            }
        }
    }
    close MF;
}
