/**
  *  \file script.h
  */
#ifndef ILD2_SCRIPT_H
#define ILD2_SCRIPT_H

#include <vector>
#include <utility>
#include <iosfwd>
#include "cpluslib/string.h"
#include "cpluslib/smartptr.h"
#include "elf.h"
#include "config.h"

class ProgramSection;

/****************************** Expressions ******************************/

class ExprNode {
 public:
    typedef std::pair<Ptr<ProgramSection>, Elf32_Addr> value_t;

    virtual ~ExprNode();
    virtual value_t eval() = 0;

    Elf32_Word evalAbsolute();
};

class ExprSymbolNode : public ExprNode {
    string_t name;
 public:
    ExprSymbolNode(const string_t& name);
    ~ExprSymbolNode();
    value_t eval();
};

class ExprIntegerNode : public ExprNode {
    Elf32_Word value;
 public:
    ExprIntegerNode(Elf32_Word value) : value(value) { }
    ~ExprIntegerNode();
    value_t eval();
};

/********************************* Script ********************************/

struct ScriptCommand {
    enum SectionAttributes {
        sa_Linear = 1
    };
    enum Action {
        a_NOP,
        
        /** Define a symbol.
            name     symbol name
            ex1      value
            info     flags (not yet used) */
        a_DefineSymbol,

        /** Set Load and Relocation address.
            ex1      value */
        a_SetLoadAddress,

        /** Set Relocation address.
            ex1      value */
        a_SetRelocationAddress,

        /** Start a section in output.
            name     name
            info     flags */
        a_StartSection,

        /** Add section from input.
            name     name wildcard
            file     file wildcard */
        a_AddSection,

        /** Set input padding. Input padding is inserted between
            sections created with a_AddSection (=sections from input).
            ex1      add
            ex2      align */
        a_SetInputPad,

        /** Set output padding. Output padding is inserted between
            sections creaded with a_StartSection (=sections in output) */
        a_SetOutputPad,

        /** Set pattern for input padding.
            ex1      value
            info     type (1, 2, 4) */
        a_SetInputPadPattern
    };

    Action         action;
    string_t       name;
    string_t       file;
    Ptr<ExprNode>  ex1, ex2;
    int            info;

    ScriptCommand(Action a, string_t name)
        : action(a), name(name), info(0) { }
    ScriptCommand()
        : action(a_NOP), info(0) { }
    ScriptCommand(Action a, string_t name, string_t file,
                  Ptr<ExprNode> ex1, Ptr<ExprNode> ex2, int info)
        : action(a), name(name), file(file), ex1(ex1), ex2(ex2),
          info(info) { }

#ifdef DEBUG
    void dump(std::ostream&);
#endif
};

class ScriptParser {
 public:
    typedef std::vector<ScriptCommand> command_vector;
    typedef command_vector::iterator   iterator;
 private:
    command_vector commands;
 public:
    ScriptParser();
    iterator begin() { return commands.begin(); }
    iterator end()   { return commands.end(); }
#ifdef DEBUG
    void dump(std::ostream&);
#endif
};

#endif
