/**
  *  \file script.cc
  */

#include "script.h"
#include "except.h"
#include "section.h"
#include "symbol.h"
#include "cpluslib/assert.h"
#include "cpluslib/misc.h"
#include "param.h"

ExprNode::~ExprNode()
{ }

Elf32_Addr
ExprNode::evalAbsolute()
{
    value_t v = eval();
    if (v.first) {
        if (!v.first->isInImage())
            throw LinkerError("Section address not known: `" + Section::getSectionName(v.first) + "'");
        return v.second + v.first->getRelocationAddress();
    } else {
        return v.second;
    }
}

ExprSymbolNode::ExprSymbolNode(const string_t& s)
    : name(s)
{
    /* FIXME: probably mark symbol undefined */
}

ExprSymbolNode::~ExprSymbolNode()
{ }

ExprSymbolNode::value_t
ExprSymbolNode::eval()
{
    Ptr<Symbol> sym = symbols.getSymbol(name);
    if(!sym.ptr() || !sym->defined)
        throw LinkerError("undefined symbol: `" + name + "'");

    return value_t(sym->section, sym->address);
}

ExprIntegerNode::~ExprIntegerNode()
{ }

ExprIntegerNode::value_t
ExprIntegerNode::eval()
{
    return value_t(0, value);
}

/******************************** Commands *******************************/

#ifdef DEBUG
static const char* command_names[] = {
    "nop",
    "DefineSymbol",
    "SetLoadAddress",
    "SetRelocationAddress",
    "StartSection",
    "AddSection",
    "SetInputPad",
    "SetOutputPad",
    "SetInputPadPattern"
};

void
ScriptCommand::dump(std::ostream& out)
{
    ASSERT(action < countof(command_names));

    out << command_names[action] << "\n";
    if (name.length())
        out << "\tname: " << name << "\n";
    if (file.length())
        out << "\tfile: " << file << "\n";
    if (ex1)
        out << "\tex1: (non-null)\n";
    if (ex2)
        out << "\tex2: (non-null)\n";
    if (info)
        out << "\tinfo: " << info << "\n";
}

void
ScriptParser::dump(std::ostream& out)
{
    for (iterator i = begin(); i != end(); ++i)
        i->dump(out);
}
#endif

/***************************** Script Parser *****************************/

static char* section_names[] = {
    ".init",
    ".text",
    ".gnu.linkonce.t",
    ".fini",
    ".rodata",
    ".gnu.linkonce.r",
    "__libc_subinit",
    ".ctors",
    ".dtors",
    ".eh_frame",
    ".gcc_except_table",
    ".data",
    ".gnu.linkonce.d",
    ".got",
    ".bss",
    0
};

/* For now, return a fixed dummy stream. */
ScriptParser::ScriptParser()
{
    const char*const* nameptr = section_names;

    int padmode = 0;

    commands.push_back(ScriptCommand(ScriptCommand::a_SetLoadAddress,
                                     string_t(), string_t(),
                                     new ExprIntegerNode(0x8040000),
                                     0, 0));

    while(*nameptr) {
        ScriptCommand c1 = ScriptCommand(ScriptCommand::a_StartSection, *nameptr);
        ScriptCommand c2 = ScriptCommand(ScriptCommand::a_AddSection,   *nameptr);

        /*
           When doing a real incremental link, these will have to get padding.
           .init / .fini: pad with NOP (0x90)
           .ctor / .dtor / __libc_subinit: pad with pointer to a dummy routine
           .gcc_except_table: pad with -1 (0xFFFFFFFF), which is "ignored" by
              libgcc2.c::find_exception_handler (end-of-this-module marker)
           .eh_frame: will get tricky. A (working) solution is to *patch*
              the dwarf information. In code: insert `.long 0's just before
              the `.LEFDE1: / .set .LLFDE1,.LEFDE1-.LSFDE1 # FDE Length Symbol'
              No idea what this really does, though ;-)
              Best(tm) solution is synthesizing a dummy record which fills
              up all space.
        */
        if(c1.name == ".init" || c1.name == ".fini"
           || c1.name == ".ctors" || c1.name == ".dtors"
           || c1.name == "__libc_subinit"
           || c1.name == ".eh_frame"
           || c1.name == ".gcc_except_table") {
            /* linear sections */
            c1.info = ScriptCommand::sa_Linear;
            if(link_mode != LM_NORMAL && !padmode) {
                /*
                  For now, in incremental mode, these sections do not
                  receive input padding. This is not yet optimal:
                  + init/fini could be padded with NOPs. However, in normal
                    gcc-compiled code, these sections never change.
                  + ctors/dtors/__libc_subinit should be padded with
                    LONG(dummy function). __libc_subinit usually never changes,
                    and the others only have to compensate for modules
                    being inserted. Problem: where to get the dummy function.
                    - private .o file
                    - crtn.o::.init/.fini
                */
                ScriptCommand ipad(ScriptCommand::a_SetInputPad, "", "",
                                   new ExprIntegerNode(0), new ExprIntegerNode(1), 0);
                ScriptCommand opad(ScriptCommand::a_SetOutputPad, "", "",
                                   new ExprIntegerNode(0), new ExprIntegerNode(4096), 0);
                commands.push_back(ipad);
                commands.push_back(opad);
            }
            padmode = 1;
        } else {
            if(link_mode != LM_NORMAL && padmode) {
                /* Add input padding for the rest */
                ScriptCommand ipad(ScriptCommand::a_SetInputPad, "", "",
                                   new ExprIntegerNode(4096), new ExprIntegerNode(4096), 0);
                ScriptCommand opad(ScriptCommand::a_SetOutputPad, "", "",
                                   new ExprIntegerNode(16384), new ExprIntegerNode(4096), 0);
                commands.push_back(ipad);
                commands.push_back(opad);
            }
            padmode = 0;
        }

        c2.file = "*";
        if(std::strncmp(*nameptr, ".gnu", 4) == 0)
            c2.name += "*";

        commands.push_back(c1);
        commands.push_back(c2);

        ++nameptr;
    }
}
