/**
  *  \file parse.cc
  */

#include <cstdio>
#include "parse.h"

/*
 *  Parse at most /max/ bytes from /n/ as a base-/base/ number
 */
uint32 parseNumberHelper(const char* n, std::size_t max, int base)
{
    uint32 ret = 0;
    for(std::size_t i = 0; i < max; ++i) {
        if(n[i] < '0' || n[i] > '9')
            break;
        ret = base*ret + n[i] - '0';
    }
    return ret;
}

/*
 *  Store a number, at most /max/ bytes
 */
void storeNumberHelper(char* n, std::size_t max, int base, uint32 num)
{
    char buffer[20];
    std::sprintf(buffer, base == 8 ? "%lo" : "%lu", (unsigned long) num);

    storeStringHelper(n, max, buffer);
}

void storeStringHelper(char* n, std::size_t max, const char* src)
{
    for(std::size_t i = 0; i < max; ++i)
        *n++ = *src ? *src++ : ' ';
}

// Quote a file name. A quoted file name does not contain '\n'
// (replaced by literal "\n"), and no unquoted occurences of '(', ')'
// and '\'. This way, they can be used to form archive member file
// names (/quoteSymbol(archive_name) + "(" + quoteFilename(member_name) + ")"/).
string_t
quoteFilename(string_t s)
{
    string_t::size_type n = 0;
    while((n = s.find_first_of("\\()\n", n)) != string_t::npos) {
        if(s[n] == '\n') {
            s.erase(n, 1);
            s.insert(n, "\\n");
        } else {
            s.insert(n, "\\");
            n += 2;
        }
    }
    return s;
}

string_t
unquoteFilename(string_t thing)
{
    string_t::size_type n = 0;
    while((n = thing.find("\\", n)) != string_t::npos) {
        thing.erase(n, 1);
        if(n >= thing.length())
            break;
        if(thing[n] == 'n')
            thing[n] = '\n';
        ++n;
    }
    return thing;
}

void
parseFilename(string_t in, string_t& filename, string_t& member)
{
    // this can only be an archive member name if it ends with a rparen
    if(in.length() && in[in.length()-1] == ')') {
        /* okay. maybe-split. */
        string_t::size_type n = 0;
        while((n = in.find_first_of("\\(", n)) != string_t::npos) {
            if(in[n] == '(') {
                /* seems to be archive with file name */
                filename = unquoteFilename(in.substr(0, n));
                member = unquoteFilename(in.substr(n+1, in.length()-n-3));
                return;
            } else {
                n += 2;
            }
        }
    }

    filename = unquoteFilename(in);
}
