/**
  *  \file output.cc
  */

#include <cstdio>
#include "output.h"
#include "address.h"
#include "except.h"
#include "cpluslib/assert.h"
#include "cpluslib/buffer.h"
#include "table.h"

DefaultOutput::DefaultOutput(string_t filename)
    : file(), filename(filename)
{ }

DefaultOutput::~DefaultOutput()
{
    if (file.isOpen()) {
        /* FIXME: is this necessary? is this good? */
        file.close();
        std::remove(filename.c_str());
    }
}

void
DefaultOutput::startOutput(Elf32_Addr start, Elf32_Addr first)
{
    header.e_entry = start;

    first_vaddr = first;
    first_offset = first % PAGE_SIZE;
    if (first_offset < sizeof(Elf32_Ehdr))
        first_offset += PAGE_SIZE;

    if (!file.create(filename.c_str(), 0600))
        throw LinkerError(filename, file.openErrorMessage());
}

/** Write output.
    \param data       data (may be NULL if data_size is 0)
    \param address    virtual address
    \param data_size  initialized data size
    \param vm_size    allocated data size
    \param prot       protection (PT_foo) */
void
DefaultOutput::writeOutput(void* data,
                           Elf32_Addr address,
                           Elf32_Word data_size,
                           Elf32_Word vm_size,
                           int prot)
{
    ASSERT(data_size <= vm_size);
    ASSERT(file.isOpen());
    if (vm_size == 0)
        return;
    
    Elf32_Phdr r = {
        PT_LOAD,                              // p_type
        address + first_offset - first_vaddr, // p_offset = file position
        address,                              // p_vaddr  = virtual address
        address,                              // p_paddr  = phys. address (unused)
        data_size,                            // p_filesz
        vm_size,                              // p_memsz
        prot,                                 // p_flags
        PAGE_SIZE                             // p_align
    };

    pheaders.insert(r);
    
    if(data_size)
        file.write(r.p_offset, data, data_size);
}

// Merge two Phdr's if possible. Returns false if merging was not possible.
// When Phdr's were merged, returns true and updates /a/. /b/ can then be
// discarded. Assumes that /b/ is the next Phdr in the list after /a/,
// sorted by starting virtual address.
static bool
maybeMerge(Elf32_Phdr& a, const Elf32_Phdr& b)
{
    /* must be correctly aligned (should always be true here) */
    if(a.p_vaddr - a.p_offset != b.p_vaddr - b.p_offset)
        return false;
    
    /* same protection */
    if(a.p_flags != b.p_flags || a.p_align != b.p_align)
        return false;

    if(a.p_filesz != a.p_memsz) {
        /* first one only partially initialized. Can only be merged
           with completely uninitialized PHdr. Possible gap inbetween
           will be zeroed by the kernel. */
        if(b.p_filesz != 0)
            return false;
        
        a.p_memsz = b.p_memsz + (b.p_vaddr - a.p_vaddr);
        return true;
    }

    /* first one completely initialized */
    if(b.p_filesz != 0)
        /* second one has data. A possible gap between these will have
           indeterminate contents (whatever happens to be in the file
           at that position), but that's not a problem: it's an error
           for the program to refer there. */
        a.p_filesz = b.p_filesz + (b.p_offset - a.p_offset);
    a.p_memsz  = b.p_memsz + (b.p_vaddr - a.p_vaddr);
    return true;
}

/** Compact the PHdr list. After writing the output, we have one PHdr
    per input section in our data structure. Linux would not have a
    problem with this, but generally, programs can go by with only 3 of
    them (text, rodata, data/bss). */
void
DefaultOutput::mergePhdrs()
{
    /* merge Phdrs, if possible */
    if(pheaders.empty())
        return;
    
    ph_set res;
    ph_set::iterator i = pheaders.begin();
    Elf32_Phdr w = *i++;

    while(i != pheaders.end()) {
        if(maybeMerge(w, *i)) {
            /* nothing */
        } else {
            res.insert(w);
            w = *i;
        }
        ++i;
    }
    res.insert(w);

    pheaders.swap(res);
}

void
DefaultOutput::endOutput()
{
    mergePhdrs();
    
    Elf32_Word highest_vaddr = first_vaddr;
    
    int n = 0;
    // FIXME: this can probably be optimized based on the fact
    // that a set is always sorted
    for(ph_set::iterator i = pheaders.begin(); i != pheaders.end(); ++i) {
        Elf32_Word a = i->p_vaddr + i->p_memsz;
        if(a > highest_vaddr)
            highest_vaddr = a;
        ++n;
    }

    /* Position of program headers in file */
    Elf32_Word ph_offs = highest_vaddr + first_offset - first_vaddr;

    VLArray<Elf32_Phdr> ph(n);
    int n1 = 0;
    for(ph_set::iterator i = pheaders.begin(); i != pheaders.end(); ++i)
        ph[n1++] = *i;

    file.write(ph_offs, ph.getBuffer(), n * sizeof(Elf32_Phdr));

    for(int i = 0; i < EI_NIDENT; ++i)
        header.e_ident[i] = 0;
    
    header.e_ident[0]          = 127;
    header.e_ident[1]          = 'E';
    header.e_ident[2]          = 'L';
    header.e_ident[3]          = 'F';
    header.e_ident[EI_CLASS]   = ELFCLASS32;
    header.e_ident[EI_DATA]    = ELFDATA2LSB;
    header.e_ident[EI_VERSION] = EV_CURRENT;

    header.e_type      = ET_EXEC;
    header.e_machine   = EM_386;
    header.e_version   = EV_CURRENT;
    //header.e_entry   = already set
    header.e_phoff     = ph_offs;
    header.e_shoff     = 0;
    header.e_flags     = 0;
    header.e_ehsize    = sizeof(Elf32_Ehdr);
    header.e_phentsize = sizeof(Elf32_Phdr);
    header.e_phnum     = n;
    header.e_shentsize = sizeof(Elf32_Shdr);
    header.e_shnum     = 0;
    header.e_shstrndx  = 0;
    file.write(0, &header, sizeof(header));

    file.chmod(File::fixMode(0755));
    file.close();
}
