/**
  *  \file object.h
  *  \brief Object files
  */
#ifndef ILD_OBJECT_H
#define ILD_OBJECT_H

#include <vector>

#include "section.h"
#include "cpluslib/smartptr.h"
#include "cpluslib/string.h"
#include "vfile.h"

class StateObjectWriter;

/* Generic object */
class LinkObject {
    LinkObject(const LinkObject&);
    LinkObject& operator=(const LinkObject&);

 protected:
    Ptr<VFile>     file;        ///< virtual file.
    string_t       name;        ///< quoted name (`file.o', `file.a(mem.o)', `c:\\file.obj').
    
 public:
    LinkObject(Ptr<VFile> file, string_t name);
    virtual ~LinkObject();

    /* Accessors */
    Ptr<VFile>  getFile() { return file; }
    string_t    getName() { return name; }

    virtual void getInformation(SymbolTable& symtab, MemoryAllocator& memalloc) = 0;
    virtual void writeStateFile(StateObjectWriter& o);
};

/* ELF object (corresponding to one physical file) */
class ElfObject : public LinkObject {
    typedef std::vector<Ptr<Section> > section_vector;
    typedef Ptr<Section> (*section_factory)(LinkObject*,const Elf32_Shdr&);

    void loadObject(section_factory);  ///< load object (section headers &c)

    Elf32_Ehdr     header;      ///< ELF header
    section_vector sections;    ///< sections
 public:
    ElfObject(Ptr<VFile> afile, string_t aname, section_factory = Section::create);

    /* Accessors */
    Ptr<Section> getSection(Elf32_Word index);
    Ptr<Section> getSectionByName(string_t aname);

    void getInformation(SymbolTable& symtab, MemoryAllocator& memalloc);
    void writeStateFile(StateObjectWriter& o);

    Elf32_Half getType() const { return header.e_type; }
};

#endif
