/**
  *  \file object.cc
  *  \brief ELF object files
  *
  *  Represents an object file (basic link granularity unit)
  */

#include "object.h"
#include "except.h"
#include "log.h"
#include "state.h"

/** \class LinkObject
    \brief Object file

    Basic linking unit. An object file contains a list of sections. */

LinkObject::LinkObject(Ptr<VFile> afile, string_t aname)
    : file(afile), name(aname)
{ }

LinkObject::~LinkObject()
{ }

void
LinkObject::writeStateFile(StateObjectWriter& o)
{
    /* FIXME: abstract? */
}

/** \class ElfObject
    \brief ELF object file

    This corresponds to a physical ELF object file. */

ElfObject::ElfObject(Ptr<VFile> afile, string_t aname, section_factory f)
    : LinkObject(afile, aname)
{
    try {
        loadObject(f);
    }
    catch(LinkerError& e) {
        throw LinkerError(aname, e.msg);
    }
}

/** Load the whole object file. Creates all section objects, initializes
    them and calls their afterCreate() to get them into a consistent
    state. */
void
ElfObject::loadObject(section_factory create)
{
    /* check file format */
    ssize_t n = file->read(0, &header, sizeof(header));
    if(n != sizeof(header) || header.e_ident[0] != 127
       || header.e_ident[1] != 'E'
       || header.e_ident[2] != 'L'
       || header.e_ident[3] != 'F')
        throw LinkerError("not an ELF file");

    if(header.e_ident[EI_CLASS] != ELFCLASS32
       || header.e_ident[EI_DATA] != ELFDATA2LSB
       || header.e_shoff == 0)
        throw LinkerError("invalid ELF file type");

    /* read section headers */
    log(LOG_DEBUG) << "loading " << name
                   << "(" << header.e_shentsize << " x " << header.e_shnum << ")"
                   << std::endl;

    Table<Elf32_Shdr> shdr(header.e_shentsize, header.e_shnum);
    file->read(header.e_shoff, shdr.getBuffer(), shdr.getSize()); // FIXME: error checking...

    for(size_t i = 0; i < shdr.getCount(); i++)
        sections.push_back(create(this, shdr[i]));

    /* read section names */
    Ptr<StringTableSection> stab
        = getSection(header.e_shstrndx).cast<StringTableSection>();
    if(!stab.ptr())
        throw LinkerError("invalid e_shstrndx");

    stab->load();
    for(size_t i = 0; i < shdr.getCount(); i++)
        sections[i]->setName(stab->getString(shdr[i].sh_name));

    for(size_t i = 0; i < shdr.getCount(); i++)
        sections[i]->afterCreate();
}

/** Get section by index. */
Ptr<Section>
ElfObject::getSection(Elf32_Word index)
{
    if(index >= sections.size())
        throw LinkerError("section index out of bounds");

    return sections[index];
}

/** Get section by name. \returns 0 if there is no such thing. */
Ptr<Section>
ElfObject::getSectionByName(string_t aname)
{
    for (unsigned i = 0; i < sections.size(); ++i)
        if (sections[i]->getName() == aname)
            return sections[i];
    return 0;
}

/** Gather information. Calls all sections' getInformation(). */
void
ElfObject::getInformation(SymbolTable& symtab, MemoryAllocator& memalloc)
{
    for(Elf32_Word i = 0; i < sections.size(); ++i)
        sections[i]->getInformation(symtab, memalloc);
}

void
ElfObject::writeStateFile(StateObjectWriter& o)
{
    for (Elf32_Word i = 0; i < sections.size(); ++i)
        sections[i]->writeStateFile(o);
}
