/**
  *  \file isize.cc
  *  \brief "size" replacement
  *
  *  This summarizes the size of an object file. In addition to the
  *  normal sizes, it displays sizes without linkonce sections
  *  (templates), which is a good indication of whether you are
  *  producing bloat or the STL :-) This one doesn't really have any
  *  relation to ild/iar, but it comes in handy.
  */

#include <iostream>
#include <vector>
#include <iomanip>
#include <fcntl.h>
#include "ar-symtab.h"
#include "except.h"
#include "elf.h"

using namespace std;

const char* cur_file;

struct Sums {
    Elf32_Word text, data, bss;

    Sums() : text(0), data(0), bss(0) { }
};

void
error(const char* n)
{
    throw LinkerError(string_t(cur_file) + ": " + n);
}

static void
scan_section_headers(File& file, Elf32_Ehdr& header, Sums data[])
{
    if(header.e_shoff == 0)
        error("No section header table present");

    Table<Elf32_Shdr> shdr(header.e_shentsize, header.e_shnum);
    file.read(header.e_shoff, shdr.getBuffer(), shdr.getSize());

    if(header.e_shstrndx >= shdr.getCount() || shdr[header.e_shstrndx].sh_type != SHT_STRTAB) {
        error("invalid string table index in ELF header");
        return;
    }

    Elf32_Shdr& strtab_shdr = shdr[header.e_shstrndx];
    VLArray<char> strtab(strtab_shdr.sh_size);
    file.read(strtab_shdr.sh_offset, strtab.getBuffer(), strtab_shdr.sh_size);

    std::vector<string_t> section_names;
    for (size_t i = 0; i < shdr.getCount(); ++i)
        section_names.push_back(strtab.getBuffer() + shdr[i].sh_name);

    for(size_t i = 0; i < shdr.getCount(); i++) {
        Elf32_Shdr& h = shdr[i];
        bool is_linkonce =
            strncmp(strtab.getBuffer() + h.sh_name, ".gnu.linkonce", 13) == 0;
        if (h.sh_flags & SHF_ALLOC) {
            switch(h.sh_type) {
             case SHT_PROGBITS:
                // if (h.sh_flags & SHF_EXECINSTR)
                if (!(h.sh_flags & SHF_WRITE))
                    data[is_linkonce].text += h.sh_size;
                else
                    data[is_linkonce].data += h.sh_size;
                break;
             case SHT_NOBITS:
                data[is_linkonce].bss += h.sh_size;
                break;
            }
        }
    }
}

static void
doSize(const char* filename)
{
    /* read header */
    File elf(filename, O_RDONLY);
    if(!elf.isOpen()) {
        std::perror(filename);
        return;
    }
    cur_file = filename;

    Elf32_Ehdr header;
    ssize_t n;
    if((n = elf.read(0, &header, sizeof(header))) != sizeof(header)
       || header.e_ident[0] != 127
       || header.e_ident[1] != 'E'
       || header.e_ident[2] != 'L'
       || header.e_ident[3] != 'F') {
        error("not an ELF file");
        return;
    }
    if(header.e_ident[EI_CLASS] != ELFCLASS32 || header.e_ident[EI_DATA] != ELFDATA2LSB) {
        error("invalid ELF class");
        return;
    }

    /* ok, looks like we can grok that file */
    Sums sums[2];
    scan_section_headers(elf, header, sums);

    static bool shown = false;
    if (!shown) {
        std::cout << "    text    data     bss     dec     hex filename\n";
        shown = true;
    }
    Elf32_Word totals[2];
    totals[0] = sums[0].text + sums[0].data + sums[0].bss;
    totals[1] = sums[1].text + sums[1].data + sums[1].bss;
    
    std::cout << setw(8) << sums[0].text + sums[1].text
              << setw(8) << sums[0].data + sums[1].data
              << setw(8) << sums[0].bss + sums[1].bss
              << setw(8) << totals[0] + totals[1]
              << setw(8) << hex << totals[0] + totals[1] << dec
              << ' ' << filename << "\n"

              << setw(8) << sums[0].text
              << setw(8) << sums[0].data
              << setw(8) << sums[0].bss
              << setw(8) << totals[0]
              << setw(8) << hex << totals[0] << dec
              << ' ' << filename << " (without templates)\n";
}

int main(int, char** argv)
{
    int rv = 0;
    while (*++argv) {
        try {
            doSize(*argv);
        }
        catch(LinkerError& e) {
            std::cerr << *argv << ": " << e.msg << "\n";
            rv = 1;
        }
    }
    return rv;
}
