/**
  *  \file iranlib.cc
  *  \brief Incremental Ranlib
  */

#include <iostream>
#include <fcntl.h>
#include "ar-symtab.h"
#include "except.h"

/*
 *  Preparation: mkdir x; cd x; ar x /usr/lib/libc.a; ar rS foo.ori *.o
 *  Testing: cp foo.ori foo.a; ranlib foo.a
 *  -> GNU ranlib: 0.320 user/0.650 real average
 *     iranlib:    0.170 user/0.300 real average
 */

static void
doRanlib(const char* fname)
{
    Ptr<File> f = new File(fname, O_RDWR);
    if (!f->isOpen())
        throw LinkerError(f->openErrorMessage());
    Ptr<VFile> vf = new VFile(f, 0, -1);
    
    Archive a(vf, true);
    Archive::Iterator it(a);
    a.getOffsetOfSymbol("x");   // this will make sure the armap is read
    a.clearArmap();
    if (it.valid())
        do {
            if (it->type == Archive::M_FILE) {
                string_t fname = it->member.getFileName(a.getLfnTab());
                maybeReadSymtab(*vf->getSubFile(it->data_pos, it->size),
                                a, it.ptr(), fname.c_str());
            }
        } while (it.next());
    a.updateArmap();
    a.compactDirectory();
    a.saveHeaders();
}

int main(int, char** argv)
{
    int rv = 0;
    while (*++argv) {
        try {
            doRanlib(*argv);
        }
        catch(LinkerError& e) {
            std::cerr << *argv << ": " << e.msg << "\n";
            rv = 1;
        }
    }
    return rv;
}
