/**
  *  \file increm.h
  */
#ifndef ILD2_INCREM_H
#define ILD2_INCREM_H

#include "vmalloc.h"
#include "object.h"
#include "output.h"
#include "state.h"
#include "got.h"

class IncrementalAllocator : public MemoryAllocator {
    Ptr<StateReader> state_reader;

    /* Vector of objects */
    typedef std::vector<Ptr<LinkObject> > objvec_type;
    /* List of sections */
    typedef std::list<Ptr<ProgramSection> > list_type;
    /* For each section name, all sections */
    typedef std::map<string_t, list_type> map_type;

    objvec_type             objects;
    map_type                sections;
    Ptr<GlobalOffsetTable>  got_section;

    /* Address computation */
    Elf32_Word input_add,  input_align;
    Elf32_Word output_add, output_align;
    Elf32_Addr first_address;

    string_t pending_section_start, prev_section;

    void addSections(string_t name, string_t file, Elf32_Addr& adr, Elf32_Addr rel_diff);
    void addPadding(Elf32_Addr from, Elf32_Addr to);
 public:
    IncrementalAllocator(string_t state_file_name);
    ~IncrementalAllocator();
    
    /* Create an `LinkObject' and add it to this memory allocator. */
    void addObject(string_t name, Ptr<VFile> file);

    /* Add a section. Called by Section->getInformation */
    void addSection(Ptr<ProgramSection> sec);

    /* Add a common symbol */
    void addCommon(string_t name, Elf32_Word size,
                   Elf32_Word align);

    /* Force creation of a GOT */
    void createGot();

    /* Get offset of a symbol in the GOT */
    Elf32_Word getGotOffset(string_t name);

    /* Compute addresses, sections, ... */
    void computeAddresses(ScriptParser& parser);

    /* Write output (open, write, close) */
    void writeOutput(Output&);

    /* Write state file */
    void writeStateFile(StateFileWriter&);
};

class IncrementalSection : public ProgramSection {
 public:
    IncrementalSection(LinkObject* o, const Elf32_Shdr& h, const string_t& name);
    void getInformation(SymbolTable& symtab, MemoryAllocator& memalloc);
    void writeOutput(Output& o);
};

class IncrementalObject : public LinkObject {
    typedef std::vector<Ptr<IncrementalSection> > section_list;
    
    Ptr<StateReader>  reader;
    Elf32_Word        index;
    section_list      sections;

    void createSections();
 public:
    IncrementalObject(Ptr<VFile> file, string_t name, Ptr<StateReader> reader);
    void getInformation(SymbolTable& symtab, MemoryAllocator& memalloc);
    void writeStateFile(StateObjectWriter& o);
};

class IncrementalOutput : public Output {
 public:
    void startOutput(Elf32_Addr start, Elf32_Addr first);
    void writeOutput(void*       data,
                     Elf32_Addr  address,
                     Elf32_Word  data_size,
                     Elf32_Word  vm_size,
                     int         prot);
    void endOutput();
};

#endif
