/**
  *  \file increm.cc
  */

#include <fcntl.h>
#include "increm.h"
#include "except.h"
#include "log.h"
#include "wildcard.h"
#include "vmalloc.h"
#include "address.h"
#include "script.h"

static const Elf32_Addr NO_ADDRESS = ~0U;

IncrementalAllocator::IncrementalAllocator(string_t state_file_name)
    : state_reader()
{
    Ptr<File> f = new File(state_file_name.c_str(), O_RDONLY);
    if (!f->isOpen())
        throw LinkerError(state_file_name + ": " + f->openErrorMessage());
    state_reader = new StateReader(new VFile(f, 0, -1), state_file_name);
}

IncrementalAllocator::~IncrementalAllocator()
{ }

void
IncrementalAllocator::addObject(string_t name, Ptr<VFile> file)
{
    Ptr<LinkObject> o;
    Elf32_Word index = state_reader->getFileIndexByName(name);
    if (index != Elf32_Word(-1) && file->getMtime() == state_reader->getFile(index).ife_mtime) {
        /* we know this file */
        log(LOG_DEBUG) << "not adding file `" << name << "'\n";
        o = new IncrementalObject(file, name, state_reader);
    } else {
        /* file is changed or new: add a standard object */
        Ptr<ElfObject> o1 = new ElfObject(file, name);
        if (o1->getType() != ET_REL)
            throw LinkerError("File `" + name + "' is not relocatable file");
        o = o1;
    }
    objects.push_back(o);
    o->getInformation(symbols, *this);
}

/* Add a section. Called by Section->getInformation */
void
IncrementalAllocator::addSection(Ptr<ProgramSection> sec)
{
    sections[sec->getName()].push_back(sec);
}

/* Add a common symbol */
void
IncrementalAllocator::addCommon(string_t name, Elf32_Word size,
                                Elf32_Word align)
{
    throw LinkerError("unsupported: addCommon('" + name + "')");
}

/* Force creation of a GOT */
void
IncrementalAllocator::createGot()
{
    throw LinkerError("unsupported: createGot()");
}

/* Get offset of a symbol in the GOT */
Elf32_Word
IncrementalAllocator::getGotOffset(string_t name)
{
    throw LinkerError("unsupported: getGotOffset()");
}

// virtual memory addresses [from,to) are padding.
void
IncrementalAllocator::addPadding(Elf32_Addr from, Elf32_Addr to)
{
    // FIXME: nothing here yet.
}

void
IncrementalAllocator::addSections(string_t name, string_t file, Elf32_Addr& adr, Elf32_Addr rel_diff)
{
    Wildcard namewild, filewild;

    /* script reader must make sure these wildcards are valid */
    ASSERT(namewild.setExpr(name) && filewild.setExpr(file));

    map_type::iterator beg = sections.lower_bound(namewild.getPrefix());
    map_type::iterator end = sections.end(); // upper_bound(namewild.getPrefix());

    for(map_type::iterator i = beg; i != end; ++i) {
        /* match section name */
        if (!namewild.match(i->first))
            continue;
        /* okay, we have a list of sections with matching names.
           Now check file names */
        list_type& s = i->second;
        for(list_type::iterator it = s.begin(); it != s.end(); ++it) {
            LinkObject* o = (*it)->getObject();
            string_t fname = o ? o->getName() : string_t("<unknown>");
            if(filewild.match(fname) && !(*it)->isVisited()) {
                /* we want this section. Compute its address */
                Elf32_Addr sec_address = NO_ADDRESS;
                if (IncrementalSection* isec = dynamic_cast<IncrementalSection*>(it->ptr())) {
                    if (isec->getAddress() >= adr)
                        sec_address = adr;
                }

                if (sec_address == NO_ADDRESS) {
                    /* do standard padding procedure */
                    if (pending_section_start.length() && prev_section.length())
                        sec_address = roundUp(adr + output_add, output_align);
                    else
                        sec_address = roundUp(adr + input_add, input_align);

                    /* align to section's constraints */
                    Elf32_Word align = (*it)->getAlignment();
                    if(align == 0)
                        align = 1;
                    if(align & (align-1))
                        throw LinkerError(Section::getSectionName(*it), "invalid alignment request (only powers of 2 allowed)");
                    sec_address = roundUp(sec_address, align);
                }

                /* finish previous section if needed. */
                if (pending_section_start.length() && prev_section.length())
                    symbols.addSymbol("__stop_" + prev_section, true, true, 0, adr + rel_diff, 0);

                pending_section_start.clear();

                /* padding before */
                addPadding(adr, sec_address);

                /* add the section */
                if (first_address == NO_ADDRESS)
                    first_address = adr;
                (*it)->setAddress(adr);
                (*it)->setRelocationAddress(adr + rel_diff);
                adr += (*it)->getSize();
                (*it)->markVisited();

                /* this was a linkonce section, so don't bother searching
                   more matching sections. Users can explicitly link in
                   particular incarnations by using appropriate wildcards. */
                if((*it)->isLinkOnce())
                    break;
            }
        }
    }
}

/* Compute addresses, sections, ... */
void
IncrementalAllocator::computeAddresses(ScriptParser& script)
{
//     buildCommonSection();
    if (got_section)
        addSection(got_section);

    Elf32_Addr link_addr = 0, rel_diff = 0;

    prev_section.clear();

    input_add    = 0;
    input_align  = 1;
    output_add   = 0;
    output_align = PAGE_SIZE;

    for (ScriptParser::iterator i = script.begin(); i != script.end(); ++i) {
        switch(i->action) {
         case ScriptCommand::a_NOP:
            break;
         case ScriptCommand::a_DefineSymbol:
            log(LOG_ERROR) << "Unsupported: a_DefineSymbol\n";
            break;
         case ScriptCommand::a_SetLoadAddress:
            {
                Elf32_Addr new_addr = i->ex1->evalAbsolute();
                if (new_addr < link_addr)
                    log(LOG_WARN) << "Attempt to lower link address has been ignored.\n";
                else
                    link_addr = new_addr;
                rel_diff = 0;
            }
            break;
         case ScriptCommand::a_SetRelocationAddress:
            rel_diff = i->ex1->evalAbsolute() - link_addr;
            break;
         case ScriptCommand::a_StartSection:
            /* Start a new section */
            pending_section_start = i->name;
            break;
         case ScriptCommand::a_AddSection:
            /* add section to image */
            addSections(i->name, i->file, link_addr, rel_diff);
            break;
         case ScriptCommand::a_SetOutputPad:
            output_add   = i->ex1->evalAbsolute();
            output_align = i->ex2->evalAbsolute();
            break;
         case ScriptCommand::a_SetInputPad:
            input_add   = i->ex1->evalAbsolute();
            input_align = i->ex2->evalAbsolute();
            break;
         case ScriptCommand::a_SetInputPadPattern:
            log(LOG_ERROR) << "unsupported: a_SetInputPadPattern\n";
            break;
        }
    }

    if(prev_section.length())
        symbols.addSymbol("__stop_" + prev_section, true, true, 0, link_addr + rel_diff, 0);
}

/* Write output (open, write, close) */
void
IncrementalAllocator::writeOutput(Output&)
{
    throw LinkerError("unsupported: writeOutput()");
}

/* Write state file */
void
IncrementalAllocator::writeStateFile(StateFileWriter&)
{
    throw LinkerError("unsupported: writeStateFile()");
}

/*************************** IncrementalObject ***************************/

IncrementalObject::IncrementalObject(Ptr<VFile> file, string_t name, Ptr<StateReader> reader)
    : LinkObject(file, name),
      reader(reader), index(reader->getFileIndexByName(name))
{
    ASSERT(index != Elf32_Word(-1));
}

/** Get information about this object. This has to define symbols,
    sections and GOT entries. When doing incremental links, we don't
    have to bother with GOT entries; they were already created in the
    previous run. */
void
IncrementalObject::getInformation(SymbolTable& symtab, MemoryAllocator& memalloc)
{
    Elf32_ild_FileEntry& file = reader->getFile(index);

    /* (un-)define symbols. Whoa! ELF state files rock :) */
    reader->getSection(file.ife_symbols)->getInformation(symtab, memalloc);

    /* sections */
    /* FIXME: this could probably be done *way* more elegant by doing
       the creation/registering in SectionListSection::getInformation(),
       but that would couple this module more closely to state.cc */
    createSections();
    for (section_list::iterator i = sections.begin(); i != sections.end(); ++i)
        (*i)->getInformation(symtab, memalloc);
}

void
IncrementalObject::createSections()
{
    if (sections.size())
        return;

    Ptr<SectionListSection> f = reader->getSection(reader->getFile(index).ife_sections).cast<SectionListSection>();
    if (!f)
        throw LinkerError("Invalid ife_sections");

    sections.reserve(f->getCount());
    for (Elf32_Word i = 0; i < f->getCount(); ++i)
        sections.push_back(new IncrementalSection(this, f->getSection(i),
                                                  f->getNameOf(i)));
}

void
IncrementalObject::writeStateFile(StateObjectWriter& o)
{
    throw LinkerError("unsupported: writeStateFile()");
}

IncrementalSection::IncrementalSection(LinkObject* o, const Elf32_Shdr& hdr,
                                       const string_t& name)
    : ProgramSection(o, hdr)
{
    setAddress(hdr.sh_addr);
    setRelocationAddress(hdr.sh_offset);
    setName(name);
    linkonce = (name.compare(0, 14, ".gnu.linkonce.", 14) == 0);
}

void
IncrementalSection::getInformation(SymbolTable& symtab, MemoryAllocator& memalloc)
{
    if (header.sh_flags & SHF_ALLOC)
        memalloc.addSection(this);
}

void
IncrementalSection::writeOutput(Output& o)
{
}
