/**
  *  \file ildd.cc
  *  \brief Simple "ldd" replacement
  *
  *  Lists parts of a program's DYNAMIC section. Unlike regular 'ldd',
  *  this does not try to execute the program.
  */

#include <iostream>
#include <string>
#include <fcntl.h>
#include "except.h"
#include "file.h"
#include "table.h"
#include "elf.h"

void
error(const char* n)
{
    throw LinkerError(n);
}

static const Elf32_Phdr*
getEntryByType(const Table<Elf32_Phdr>& table, Elf32_Word type)
{
    for (std::size_t i = 0; i < table.getCount(); ++i)
        if (table[i].p_type == type)
            return &table[i];
    return 0;
}

static const Elf32_Dyn*
getEntryByType(const VLArray<Elf32_Dyn>& table, Elf32_Sword type)
{
    for (std::size_t i = 0; i < table.getSize(); ++i)
        if (table[i].d_tag == type)
            return &table[i];
    return 0;
}

static std::string
getString(const VLArray<char>& a, std::size_t index)
{
    if (index >= a.getSize())
        return std::string();

    std::size_t end = index;
    while (end < a.getSize() && a[end] != 0)
        ++end;

    return std::string(a.getBuffer() + index, end - index);
}

/** Load part of the file's virtual memory.
    \param data  Output
    \param phdr  Program header
    \param file  File
    \param start Starting virtual address
    \param size  Number of bytes to load */
static void
loadVirtual(VLArray<char>& data,
            const Table<Elf32_Phdr>& phdr, File& file,
            Elf32_Addr start, Elf32_Addr size)
{
    /* clear */
    for (std::size_t i = 0; i < data.getSize(); ++i)
        data[i] = 0;

    /* read */
    Elf32_Addr end = start + size;
    for (std::size_t i = 0; i < phdr.getCount(); ++i) {
        if (phdr[i].p_type == PT_LOAD) {
            /* is it part of the region we need? */
            Elf32_Addr fstart = phdr[i].p_vaddr;
            Elf32_Addr fend   = fstart + phdr[i].p_filesz;
            if (fstart < start)
                fstart = start;
            if (fend > end)
                fend = end;
            /* yes, read it */
            if (fstart < fend) {
                Elf32_Addr ofs = fstart - start;
                Elf32_Addr num = fend - fstart;
                if (file.read(phdr[i].p_offset + fstart - phdr[i].p_vaddr,
                              data.getBuffer() + ofs, num) != num)
                    error("Unable to read virtual memory image");
            }
        }
    }
}

static void
parseDynamic(const Table<Elf32_Phdr>& phdr, const VLArray<Elf32_Dyn>& dyn, File& file)
{
    /* read string table */
    const Elf32_Dyn* strtab = getEntryByType(dyn, DT_STRTAB);
    const Elf32_Dyn* strsz  = getEntryByType(dyn, DT_STRSZ);
    VLArray<char> strings(0);
    if (strtab && strsz) {
        strings.changeSize(strsz->d_un.d_ptr);
        loadVirtual(strings, phdr, file, strtab->d_un.d_ptr, strsz->d_un.d_ptr);
    }

    /* list DT_NEEDED */
    for (std::size_t i = 0; i < phdr.getCount(); ++i) {
        if (dyn[i].d_tag == DT_NEEDED) {
            std::cout << "\tLibrary => " << getString(strings, dyn[i].d_un.d_ptr) << "\n";
        } else if (dyn[i].d_tag == DT_RPATH) {
            std::cout << "\tSearch-Path => " << getString(strings, dyn[i].d_un.d_ptr) << "\n";
        } else if (dyn[i].d_tag == DT_SONAME) {
            std::cout << "\tName => " << getString(strings, dyn[i].d_un.d_ptr) << "\n";
        }
    }
}

static int
doLDD(const char* filename)
{
    /* read header */
    File elf(filename, O_RDONLY);
    if(!elf.isOpen()) {
        std::perror(filename);
        return 1;
    }

    Elf32_Ehdr header;
    ssize_t n;
    if((n = elf.read(0, &header, sizeof(header))) != sizeof(header)
       || header.e_ident[0] != 127
       || header.e_ident[1] != 'E'
       || header.e_ident[2] != 'L'
       || header.e_ident[3] != 'F') {
        error("not an ELF file");
        return 1;
    }
    if(header.e_ident[EI_CLASS] != ELFCLASS32 || header.e_ident[EI_DATA] != ELFDATA2LSB) {
        error("invalid ELF class");
        return 1;
    }

    /* find program table */
    if (header.e_phoff == 0)
        error("No program header present");

    Table<Elf32_Phdr> phdr(header.e_phentsize, header.e_phnum);
    elf.read(header.e_phoff, phdr.getBuffer(), phdr.getSize());

    std::cout << filename << ":\n";
    const Elf32_Phdr* interp = getEntryByType(phdr, PT_INTERP);
    if (interp) {
        VLArray<char> name(interp->p_filesz);
        if (elf.read(interp->p_offset, name.getBuffer(), interp->p_filesz) != interp->p_filesz) {
            std::cout << "\tInterpreter => unreadable\n";
        } else {
            std::cout << "\tInterpreter => "
                      << getString(name, 0)
                      << "\n";
        }
    } else {
        std::cout << "\tInterpreter => none\n";
    }

    /* find DYNAMIC section */
    const Elf32_Phdr* dynamic = getEntryByType(phdr, PT_DYNAMIC);
    if (dynamic) {
        VLArray<Elf32_Dyn> dyn(dynamic->p_filesz / sizeof(Elf32_Dyn));
        std::size_t to_read = dyn.getSize() * sizeof(Elf32_Dyn);
        if (elf.read(dynamic->p_offset, dyn.getBuffer(), to_read) != to_read)
            error("Unable to read DYNAMIC section");

        parseDynamic(phdr, dyn, elf);
    }

    return 0;
}

int
main(int argc, char** argv)
{
    int rv = 0;
    for (int i = 1; argv[i] != 0; ++i) {
        try {
            rv |= doLDD(argv[i]);
        }
        catch(LinkerError& e) {
            std::cerr << argv[i] << ": " << e.msg << "\n";
            rv |= 1;
        }
    }
    return rv;
}
