/**
  *  \file ild.cc
  *
  *  Incremental Linker -- Main Module
  *
  *  This one does option parsing and general control stuff.
  */
#include <vector>
#include <string>
#include <cstdlib>
#include <algorithm>
#include <utility>
#include <cstring>
#include <cerrno>
#include <fstream>
#include <iostream>
#include <fcntl.h>
#include "file.h"
#include "log.h"
#include "archive.h"
#include "getopt.h"
#include "param.h"
#include "cpluslib/string.h"
#include "vfile.h"
#include "config.h"
#include "symbol.h"
#include "except.h"
#include "parse.h"
#include "vmalloc.h"
#include "script.h"
#include "output.h"
#include "elfio.h"
#include "state.h"
#include "increm.h"

/** Program version */
#define VERSION "0.0.1"

/** Program name for error reporting */
const char* prog_name;

/** Value of `--state-file' option, if given */
static const char* state_file_override;

/** the memory allocation strategy */
Ptr<MemoryAllocator> vm_allocator;

/** link job type */
typedef std::pair<Ptr<VFile>, string_t> linkjob_entry;
typedef std::vector<linkjob_entry> linkjob_type;

//! Exit with error message, returning errorlevel 1.
static void
fail(string_t s)
{
    std::cerr << prog_name << ": " << s << std::endl;
    std::exit(1);
}

//! Show help message and exit.
static void
showHelp()
{
    std::cout <<
        "Incremental Linker -- Version " VERSION "\n"
        "\n"
        "Usage: " << prog_name << " [-options] objects-and-libs\n"
        "\n"
        "Options:\n"
        " -Lpath\tset library path\n"
        " -lfoo\tlink with library libfoo.a\n"
        " -h\thelp\n"
        " -V\tshow version\n"
        "\n"
        "Report bugs to <streu@gmx.de>\n";
    std::exit(0);
}

//! Show version and exit.
static void
showVersion()
{
    std::cout << "Incremental Linker -- Version " VERSION "\n";
    std::exit(0);
}

// /**
//   *  Search a file using the library path. Returns an opened Ptr<File>,
//   *  or null if not found.
//   *  \param basename name of file to search
//   *  \param name_ret if non-null, receives name of file opened
//   */
// static Ptr<File>
// searchFile(string_t basename, string_t* name_ret)
// {
//     Ptr<File> f = new File;
//     for(std::vector<string_t>::const_iterator i = lib_path.begin();
//         i != lib_path.end(); ++i) {
//         string_t name = *i + "/" + basename;
//         if(f->open(name.c_str(), O_RDONLY)) {
//             if(name_ret)
//                 *name_ret = name;
//             return f;
//         }
//         if(f->getOpenErrno() != ENOENT)
//             // this is to help diagnosing permission errors. Not all
//             // `make install's install libraries with the correct permissions
//             // at the first attempt, but gld's message `No such file or
//             // directory' is more confusing than it helps for this case...
//             log(LOG_WARN) << name << ": " << f->openErrorMessage() << std::endl;
//     }
//     return 0;
// }

//! Attempt to open a library via the library path.
static bool
tryCompleteLinkjobEntry(linkjob_entry& ent)
{
    Ptr<File> f = new File;
    log(LOG_DEBUG) << "Searching library " << ent.second << std::endl;
    for(std::vector<string_t>::const_iterator i = lib_path.begin();
        i != lib_path.end(); ++i) {
        string_t name = *i + "/lib" + ent.second + ".a";
        log(LOG_DEBUG) << "  trying to open " << name << std::endl;
        if(f->open(name.c_str(), O_RDONLY)) {
            ent.second = name;
            ent.first = new VFile(f, 0, -1);
            return true;
        }
        if(f->getOpenErrno() != ENOENT)
            log(LOG_WARN) << name << ": " << f->openErrorMessage() << std::endl;
    }
    return false;
}

// Complete the specified link job, by searching for libraries.
static bool
searchLibraries(std::vector<linkjob_entry>& files)
{
    bool empty = true, error = false;
    for(std::vector<linkjob_entry>::iterator i = files.begin(); i != files.end(); ++i) {
        if(i->first.ptr()) {
            empty = false;
        } else {
            if(tryCompleteLinkjobEntry(*i)) {
                empty = false;
            } else {
                log(LOG_ERROR) << i->second << ": no matching library found\n";
                error = true;
            }
        }
    }

    if(empty)
        fail("No input files");

    return error;
}

//! Add a library to the image
static void
addLibrary(linkjob_entry& lib)
{
    Archive arch(lib.first, true);
    //vm_allocator->addLibrary(arch, lib.second);
    
    string_t sym;
    if(symbols.getFirstUndefinedSymbol(sym)) {
        do {
            off_t spos = arch.getOffsetOfSymbol(sym);
            if(spos) {
                /* armap says this archive defines that symbol */
                Archive::MPtr mem = arch.getMemberByOffset(spos);
                if(!mem.ptr()) {
                    symbols.finishSymbolIteration();
                    throw LinkerError(lib.second, "invalid armap (bad pointer)");
                }
                Ptr<VFile> memf = arch.getFile()->getSubFile(mem->data_pos, mem->size);
                memf->setMtime(parseNumber(mem->member.mtime, 10));
                string_t memn = quoteFilename(lib.second) + "(" + quoteFilename(mem->member.getFileName(arch.getLfnTab())) + ")";
                vm_allocator->addObject(memn, memf);
                if(!symbols.getSymbol(sym)->defined) {
                    symbols.finishSymbolIteration();
                    throw LinkerError(memn, "promised to define `" + sym + "' but didn't");
                }
            }
        } while(symbols.getNextUndefinedSymbol(sym));
    }
}

//! Main linker routine. This does a complete linking run.
// \throws LinkerError when linking fails.
static void
doLink(std::vector<linkjob_entry>& files, ScriptParser& parser, Output& outp)
{
    char header[4];
    std::vector<linkjob_entry>::size_type i;

    /* read all files */
    for(i = 0; i < files.size(); ++i) {
        if(files[i].first->read(0, header, 4) != 4) {
            log(LOG_ERROR) << "invalid file format: " << files[i].second << std::endl;
        } else if(std::strncmp(header, "\x7F""ELF", 4)==0) {
            files[i].first->setMtime();
            vm_allocator->addObject(quoteFilename(files[i].second), files[i].first);
        } else if(std::strncmp(header, "!<ar", 4)==0) {
            addLibrary(files[i]);
        } else {
            log(LOG_ERROR) << "invalid file format: " << files[i].second << std::endl;
        }
    }

    /* allocate addresses */
    symbols.addSymbol(start_symbol, false, false, 0, 0, 0);
    vm_allocator->computeAddresses(parser);

    if(symbols.checkUndefinedSymbols()) {
        log(LOG_ERROR) << "Linking aborted due to undefined symbols." << std::endl;
        return;
    }

    /* write output */
    vm_allocator->writeOutput(outp);

    /* write state file */
    if (state_file_p) {
        Ptr<File> f = new File();
        if(!f->create(state_file.c_str(), File::fixMode(0644))) {
            log(LOG_ERROR) << "Can't create state file.\n";
        } else {
            StateFileWriter writer(new VFile(f, 0, -1));
            vm_allocator->writeStateFile(writer);
            writer.write();
        }
    }
}

void
setLinkMode(const char* mode)
{
    std::size_t len = std::strlen(mode);
    if(len > 0) {
        if(std::strncmp(mode, "incremental", len) == 0) {
            link_mode = LM_INCREM;
            return;
        } else if(std::strncmp(mode, "normal", len) == 0) {
            link_mode = LM_NORMAL;
            return;
        }
    }
    fail("Invalid link mode specified");    
}

static void
setStateFileName()
{
    if(state_file_override) {
        /* override given */
        state_file = state_file_override;
    } else {
        /* if no state file name specified, guess one */
        state_file = output_file;
        string_t::size_type n = state_file.find_last_of(CONF_PATHSEP ".");
        if(n != string_t::npos)
            state_file.erase(n);
        state_file += ".state";
    }
    if(state_file == output_file) {
        log(LOG_ERROR) << "Output file and state file name both set to `" << state_file << "'.\n";
        fail("Please change either one using `-o' or `--state-file'");
    }
    log(LOG_INF) << "State file is `" << state_file << "'\n";
}

static Option option_defs[] = {
    { 'l', "library",       true,  0 },
    { 'L', "library-path",  true,  0 },
    { 'h', "help",          false, 0 },
    { 'V', "version",       false, 0 },
    { 'o', "output",        true,  0 },
    { 0,   "mode",          true,  OPTION(1) },
    { 0,   "state-file",    true,  OPTION(2) },
    { 0,   "no-state-file", false, OPTION(3) },
    { 0,   0,               0,     0 }
};

// Main program
int main(int /*argc*/, char** argv)
{
    const char* ldpath;
    std::vector<linkjob_entry> files;
    bool error = false;

    /* initialize /lib_path/ */
    lib_path.push_back("/lib");
    lib_path.push_back("/usr/lib");
    lib_path.push_back("/usr/local/lib");
    ldpath = getenv("ILD_LIBRARY_PATH");
    if(!ldpath)
        ldpath = getenv("LD_LIBRARY_PATH");
    if(ldpath)
        addLibPath(ldpath);

    /* parse parameters */
    prog_name = argv[0];
    try {
        for(Option::iterator i = Option::iterator(argv, option_defs); i.hasMoreElements(); ++i) {
            switch(i.id) {
             case 'L':
                addLibDir(i.arg);
                break;
             case 'l':
                files.push_back(linkjob_entry(0, string_t(i.arg)));
                break;
             case 'o':
                output_file = i.arg;
                break;
             case 'h':
                showHelp();
                break;
             case 'V':
                showVersion();
                break;
             case OPTION(1):
                setLinkMode(i.arg);
                break;
             case OPTION(2):
                state_file_override = i.arg;
                break;
             case OPTION(3):
                state_file_p = false;
                break;
             default:
                Ptr<File> f = new File(i.arg, O_RDONLY);
                if(!f->isOpen()) {
                    log(LOG_ERROR) << i.arg << ": " << f->openErrorMessage() << std::endl;
                    error = true;
                } else {
                    files.push_back(linkjob_entry(new VFile(f, 0, -1),
                                                  string_t(i.arg)));
                }
            }
        }
    }
    catch(Option::Error& e) {
        fail(e.msg);
    }

    /* Postprocess parameters */
    if(searchLibraries(files))
        error = true;
    if(error)
        fail("Linking not done.");

    setStateFileName();

    /* Do linking */
    ScriptParser parser;

    if(link_mode == LM_INCREM) {
        /* incremental link requested */
        try {
//             Ptr<File> f = new File(state_file.c_str(), O_RDONLY);
//             if (!f->isOpen())
//                 throw LinkerError("Unable to open " + state_file + ": " + f->openErrorMessage());
//             Ptr<VFile> vf = new VFile(f, 0, -1);
//             Ptr<StateReader> r = new StateReader(vf, state_file);
//             IncrementalOutput outp(output_file);
            DefaultOutput outp(output_file); /* FIXME */
            symbols.clear();
            vm_allocator = new IncrementalAllocator(state_file);
            doLink(files, parser, outp);
        }
        catch(LinkerError& e) {
            log(LOG_WARN) << "Error during incremental link: " << e.msg << std::endl;
            log(LOG_WARN) << "Trying normal link.\n";
            log(LOG_WARN) << "Nah, bailing out instead.  FIXME\n";
            //link_mode = LM_INCREM_FIRST;
        }
    }
    if(link_mode != LM_INCREM) {
        /* non-incremental, or first */
        try {
            DefaultOutput outp(output_file);
            symbols.clear();
            vm_allocator = new DefaultMemoryAllocator(/*symbols*/);
            doLink(files, parser, outp);
        }
        catch(LinkerError& e) {
            log(LOG_ERROR) << e.msg << std::endl;
            return 1;
        }
    }

    /* postprocess */
//     if(wantLog(LOG_WARN))
//         vm_allocator->warnUnlinkedSections();
    if(wantLog(LOG_INF)) {
        std::cerr << "Total bytes read:    " << File::bytes_read << std::endl
                  << "Total bytes written: " << File::bytes_written << std::endl;
    }
}
