/*
 *  Incremental `AR'
 */
#include <vector>
#include <string>
#include <iostream>
#include <cstdlib>
#include <exception>
#include <csignal>
#include <fcntl.h>
#include "log.h"
#include "iar.h"
#include "iar-ptx.h"
#include "iar-d.h"
#include "iar-r.h"
#include "cpluslib/string.h"
#include "except.h"

#define VERSION "0.0.1"

enum Operation {
    O_UNSPECIFIED,
    O_DELETE,                   // ar d
    O_PRINT,                    // ar p
    O_ADD,                      // ar r; ar q
    O_LIST,                     // ar t
    O_EXTRACT                   // ar x
};

bool verbose_flag = false;      // ar -v
bool create_flag = false;       // ar -c (create/recreate)
bool truncate_flag = false;     // ar -f (truncate names)
bool preserve_flag = false;     // ar -o (preserve timestamps on extract)
bool no_ranlib_flag = false;    // ar -S (no ranlib)

std::vector<const char*> file_names;
const char* arch_name = 0;
const char* prog_name;

Operation operation = O_UNSPECIFIED;

extern "C"
{
    typedef void (*sighandler_t)(int);
}

static void catchSignalsTo(sighandler_t f)
{
    std::signal(SIGINT, f);
    std::signal(SIGTERM, f);
#ifdef SIGQUIT
    std::signal(SIGQUIT, f);
#endif
#ifdef SIGPIPE
    std::signal(SIGPIPE, f);
#endif
}

static void fail(string_t s)
{
    std::cerr << prog_name << ": " << s << std::endl;
    std::exit(1);
}

static void showVersion()
{
    std::cout << "Incremental AR -- Version " VERSION << std::endl;
    std::exit(0);
}

static void showHelp()
{
    std::cout << "Incremental AR -- Version " VERSION << std::endl
              << std::endl;
    std::cout <<
        "Usage: " << prog_name << " [-]command[options] archive.a [members.o ...]\n"
        "\n"
        "Commands:\n"
        "  d\tDelete\n"
        "  p\tPrint (extract to stdout)\n"
        "  r, q\tAdd (replace)\n"
        "  t\tList content\n"
        "  x\tExtract\n"
        "\n"
        "Options:\n"
        "  -c\tCreate (commands r, q)\n"
        "  -f\tTruncate filenames (commands r, q)\n"
        "  -o\tPreserve timestamps (command x)\n"
        "  -v\tVerbose\n"
        "  -V\tShow Version\n"
        "\n"
        "Report bugs to <streu@gmx.de>\n";
    std::exit(1);
}

static void setOperation(Operation o)
{
    if(operation != O_UNSPECIFIED)
        fail("Multiple commands specified");
    operation = o;
}    

int main(int argc, char** argv)
{
    prog_name = argv[0];
    log_level = LOG_INF;

    if(argc == 1)
        showHelp();
    
    /* Parse arguments */
    for(int i = 1; i < argc; ++i) {
        if(i == 1 || argv[i][0] == '-') {
            const char* p = argv[i];
            if(*p == '-')
                ++p;
            while(*p) {
                switch(*p) {
                 case 'd':
                    setOperation(O_DELETE);
                    break;
                 case 'm':
                    fail("The `m' operation (move) is not supported");
                    break;
                 case 'p':
                    setOperation(O_PRINT);
                    break;
                 case 'q':
                 case 'r':
                    setOperation(O_ADD);
                    break;
                 case 't':
                    setOperation(O_LIST);
                    break;
                 case 'x':
                    setOperation(O_EXTRACT);
                    break;
                 case 'a':
                 case 'b':
                 case 'i':
                    fail(string_t("The `-") + *p + "' option is not supported");
                    break;
                 case 'c':
                    create_flag = true;
                    break;
                 case 'f':
                    truncate_flag = true;
                    break;
                 case 'l':
                 case 'u':
                    break;
                 case 'o':
                    preserve_flag = true;
                    break;
                 case 'S':
                    no_ranlib_flag = true;
                    break;
                 case 's':
                    log(LOG_INF) << "The `-" << *p << "' option was ignored.\n";
                    break;
                 case 'v':
                    verbose_flag = true;
                    break;
                 case 'V':
                    showVersion();
                    break;
                 case 'h':
                    showHelp();
                    break;
                 default:
                    fail(string_t("Invalid option or command `") + *p + "'");
                    break;
                }
                ++p;
            }
        } else {
            if(!arch_name)
                arch_name = argv[i];
            else
                file_names.push_back(argv[i]);
        }
    }

    /* Verify arguments */
    if(operation == O_UNSPECIFIED)
        fail("You did not specify an operation (one of `dprtqx')");
    if(!arch_name)
        fail("You did not specify an archive name");
    if(preserve_flag && operation != O_EXTRACT)
        log(LOG_INF) << "The `-o' option only makes sense with the `x' operation (extract).\n";
    if(create_flag && operation != O_ADD)
        log(LOG_INF) << "The `-c' option only makes sense with the `r' or `q' operation (add).\n";

    /* Go! */
    try {
        switch(operation) {
         case O_DELETE:
            catchSignalsTo(SIG_IGN);
            iarDeleteArchiveMembers();
            break;
         case O_PRINT:
            iarIterateInOrder(call(iarPrintArchiveMember), O_RDONLY, false);
            break;
         case O_ADD:
            catchSignalsTo(SIG_IGN);
            iarAddArchiveMembers();
            break;
         case O_LIST:
            iarIterateInOrder(call(iarListArchiveMember), O_RDONLY, false);
            break;
         case O_EXTRACT:
            iarIterateInOrder(call(iarExtractMember), O_RDONLY, false);
            break;
         default:
            /* silence compiler warning */
            ;
        }
    }
    catch(LinkerError& e) {
        std::cerr << prog_name << ": " << e.msg << std::endl;
        std::exit(1);
    }
    catch(std::exception& e) {
        std::cerr << prog_name << ": " << e.what() << std::endl;
        std::exit(1);
    }
}
