#define _POSIX_SOURCE           // stat
#include <iostream>
#include <cstdio>
#include <vector>
#include <cstring>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "iar-r.h"
#include "file.h"
#include "iar.h"
#include "archive.h"
#include "vfile.h"
#include "elf.h"
#include "section.h"
#include "log.h"
#include "except.h"
#include "parse.h"
#include "cpluslib/assert.h"
#include "ar-symtab.h"

//
//  `$AR cr libc.a *.o' (*.o == libc-5.4.44, 1 MB)
//  GNU ar, archive already exists (same content):  2.1 real (1.0 user)
//  GNU ar, archive does not exist:                 1.3 real (0.9 user)
//  iar, both cases:                                0.8 real (0.4 user)
//
//  `$AR cr ild.a *.o' (*.o == ild 11/Mar/2001, 9 MB)
//  GNU ar: 4.5 real (0.8 user)
//  iar:    3.5 real (0.05 user)
//
//  `$AR r ild.a *.o' (*.o == ild 11/Mar/2001, 9 MB)
//  GNU ar:         4.7 real   (0.7 user)
//  iar:            0.051 real (0.050 user)
//  GNU ar with -u: 0.025 real (0.010 user)
//

static void copyFile(File& in, Ptr<VFile> out, off_t out_pos, off_t size)
{
    enum { BUFSIZE = 2 << 16 };
    VLArray<char> b(BUFSIZE);
    off_t in_pos = 0;

    while(size) {
        off_t now = size < BUFSIZE ? size : BUFSIZE;
        if(in.read(in_pos, b.getBuffer(), now) != now)
            throw LinkerError("Read error");
        if(out->write(out_pos, b.getBuffer(), now) != now)
            throw LinkerError("Write error");
        in_pos += now;
        out_pos += now;
        size -= now;
    }
}

static const char* getBasename(const char* name)
{
    const char* basename = std::strrchr(name, '/');
    if(basename)
        ++basename;
    else
        basename = name;
    return basename;
}

static void addToArchive(Archive& arch, const char* name, string_t& lfntab)
{
    Ptr<File> in = new File(name, O_RDONLY);
    if(!in->isOpen()) {
        std::cerr << name << ": " << in->openErrorMessage() << std::endl;
        return;
    }

    struct stat b;
    if(in->fstat(b) != 0) {
        std::perror(name);
        return;
    }
    if(!S_ISREG(b.st_mode)) {
        log(LOG_WARN) << name << ": not a regular file.\n";
        return;
    }

    Archive::MPtr new_mem = arch.allocateSpace(b.st_size);

    /* set up member record */
    const char* basename = getBasename(name);
    
    if(strlen(basename) <= 15 || truncate_flag) {
        /* fits in filename field */
        char buf[20];
        std::sprintf(buf, "%.15s/", basename);
        storeString(new_mem->member.filename, buf);
    } else {
        /* need long name */
        string_t what = string_t(basename) + "/\n";
        string_t::size_type i = lfntab.find(what);
        if(i == string_t::npos) {
            i = lfntab.length();
            lfntab += what;
        }
        char buf[20];
        std::sprintf(buf, "/%ud", (unsigned int) i);
        storeString(new_mem->member.filename, buf);
    }

    storeNumber(new_mem->member.mtime, 10, b.st_mtime);
    storeNumber(new_mem->member.uid,   10, b.st_uid);
    storeNumber(new_mem->member.gid,   10, b.st_gid);
    storeNumber(new_mem->member.mode,  8, b.st_mode);

    arch.maybeBumpMtime(b.st_mtime);

    copyFile(*in, arch.getFile(), new_mem->data_pos, b.st_size);

    if (!no_ranlib_flag) {
        VFile vin(in, 0, -1);
        maybeReadSymtab(vin, arch, new_mem, name);
    }
}

void iarAddArchiveMembers()
{
    if(file_names.empty()) {
        std::cerr << "No member file name specified\n";
        return;
    }

    /* open archive */
    Ptr<File> f = new File();

    if(!create_flag) {
        if(!f->open(arch_name, O_RDWR)) {
            log(LOG_WARN) << "`" << arch_name << "' does not exist -- creating it.\n";
            create_flag = true;
        }
    }

    if(create_flag)
        if(!f->create(arch_name, 0666))
            throw LinkerError(string_t("Can't create `") + arch_name + "': " +
                              f->openErrorMessage());

    Ptr<Archive> arch;
    string_t lfntab;
    std::vector<bool> add_this_file, replaced_this_file;

    if(create_flag) {
        /* creating an archive */
        arch = new Archive(new VFile(f, 0, -1), 42);

        /* approximate size of headers */
        unsigned factor = no_ranlib_flag ? 16 : 512;
        Archive::MPtr headers = arch->allocateSpace(factor * file_names.size());
        headers->type = Archive::M_ARMAP;
    } else {
        /* updating existing archive */
        arch = new Archive(new VFile(f, 0, -1), true);
        add_this_file.reserve(file_names.size());
        replaced_this_file.reserve(file_names.size());

        /* delete existing members */
        for(std::vector<const char*>::iterator fn_iter = file_names.begin();
            fn_iter != file_names.end(); ++fn_iter)
        {
            struct stat s;
            bool add_it = false, replaced_it = false;
            if(stat(*fn_iter, &s) != 0)
                perror(*fn_iter);
            else if(!S_ISREG(s.st_mode))
                std::cerr << *fn_iter << ": not a regular file\n";
            else {
                /* file was stat'able */
                Archive::MPtr mptr = arch->getMemberByName(getBasename(*fn_iter));
                if(mptr.ptr()) {
                    if(parseNumber(mptr->member.mtime, 10) < s.st_mtime) {
                        /* we should add it */
                        mptr->type = Archive::M_PAD;
                        mptr->dirty = true;
                        // FIXME: delete it from by_name(?)
                        arch->changeArmapEntry(mptr->pos, 0);
                        add_it = true;
                        replaced_it = true;
                    }
                } else
                    add_it = true;
            }
            add_this_file.push_back(add_it);
            replaced_this_file.push_back(replaced_it);
        }
        ASSERT(add_this_file.size() == file_names.size());
        arch->compactDirectory();
        arch->updateArmap();
        lfntab = string_t(arch->getLfnTab().getBuffer(),
                          arch->getLfnTab().getSize());
        arch->selfcheck();
    }

    /* now add members */
    for(std::vector<const char*>::size_type i = 0; i < file_names.size(); ++i)
        if(create_flag || add_this_file[i]) {
            if(verbose_flag)
                if (create_flag || !replaced_this_file[i])
                    std::cout << "a - " << file_names[i] << "\n";
                else
                    std::cout << "r - " << file_names[i] << "\n";
            addToArchive(*arch, file_names[i], lfntab);
        }

    if(lfntab.length() != arch->getLfnTab().getSize()) {
        ASSERT(lfntab.length() > arch->getLfnTab().getSize());
        arch->setLfnTab(lfntab.data(), lfntab.length());
    }

    /* update archive */
    arch->updateArmap();
    arch->compactDirectory();
    arch->saveHeaders();
}
