/*
 *  `iar p', `iar t' and `iar x' commands
 *
 *  These are non-modifying commands which require iteration through
 *  the archive in the order specified on the command line.
 */
#include <algorithm>
#include <vector>
#include <iomanip>
#include <cstdio>
#include <iostream>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <utime.h>
#include <unistd.h>
#include "iar-ptx.h"
#include "vfile.h"
#include "iar.h"
#include "log.h"
#include "closure.h"
#include "parse.h"
#include "except.h"

/* format a number-of-seconds value into string for output */
static string_t timeString(uint32 time)
{
    time_t t = time;
#if 0
    /* This works, of course (and is IMHO very clean)... */
    char timestr[100];
    strftime(timestr, sizeof(timestr),
             "%b %d %H:%M %Y",
             localtime(&t));
    return timestr;
#else
    /* ... but this one makes `iar's output 100% bit-for-bit
       compatible with GNU `ar'. Maybe someone cares. */
    char* ctime_res = ctime(&t);
    return string_t(ctime_res + 4, 12) + string_t(ctime_res + 19, 5);
#endif
}

/*
 *  Copy data from VFile /v/, offset /start/, length /length/, to
 *  the file descriptor /fd/.
 */
void copyToHandle(VFile& v, off_t start, off_t length, int fd)
{
    static char buffer[10240];

    while(length > 0) {
        off_t now = std::min(off_t(sizeof(buffer)), length);
        off_t got = v.read(start, buffer, now);
        start += now;
        length -= now;
        if(got != now)
            log(LOG_ERROR) << "premature end of file\n";

        off_t written = ::write(fd, buffer, got);
        if(got != written)
            log(LOG_ERROR) << "write error\n";
    }
}

/*
 *  Print one archive member to stdout
 */
void iarPrintArchiveMember(Archive& arch, Archive::MemberData& mem)
{
    if(mem.type != Archive::M_FILE)
        return;

    if(verbose_flag)
        std::cout << "\n<member " << mem.member.getFileName(arch.getLfnTab()) << ">\n\n" << std::flush;
    copyToHandle(*arch.getFile(),
                 mem.data_pos, mem.size, STDOUT_FILENO);
}

/*
 *  List archive member
 */
void iarListArchiveMember(Archive& arch, Archive::MemberData& mem)
{
    if(mem.type != Archive::M_FILE)
        return;

    string_t name = mem.member.getFileName(arch.getLfnTab());
    if(verbose_flag) {
        uint32 size  = mem.size;
        uint32 uid   = parseNumber(mem.member.uid, 10);
        uint32 gid   = parseNumber(mem.member.gid, 10);
        uint32 mode  = parseNumber(mem.member.mode, 8);
        uint32 mtime = parseNumber(mem.member.mtime, 10);

        for(int i = 8; i >= 0; i--)
            std::cout << char((mode & (1 << i)) ? "xwrxwrxwr"[i] : '-');
        std::cout << " " << uid << "/" << gid
                  << " " << std::setw(6) << size << " "
                  << timeString(mtime) << " " << name << std::endl;
    } else {
        std::cout << name << std::endl;
    }
}

/*
 *  Member extraction: `time $AR x /usr/lib/libc.a vm86.o' (591th member
 *  of 651)
 *    New Implementation (unord.):  0.073 real
 *    New Implementation (ordered): 0.064 real
 *    Old Implementation:           0.036 real
 *    GNU ar:                       0.200 real
 *  `time $AR x /usr/lib/libc.a assert.o' (2nd member in file)
 *    New Implementation (ordered): 0.024 real
 *    GNU ar:                       0.200 real
 *  `time $AR x /usr/lib/libc.a' (651 members, ~1 Meg)
 *    New Implementation (unord.):  0.255 real (0.060 user)
 *    New Implementation (ordered): 0.255 real (0.050 user)
 *    Old Implementation:           0.200 real (0.020 user)
 *    GNU ar:                       0.560 real
 */
void iarExtractMember(Archive& arch, Archive::MemberData& mem)
{
    if(mem.type != Archive::M_FILE)
        return;

    string_t name = mem.member.getFileName(arch.getLfnTab());

    if(verbose_flag)
        std::cout << "x - " << name << std::endl;

    int mode = parseNumber(mem.member.mode, 8) & 0777;
    int fd = creat(name.c_str(), mode);
    if(fd < 0) {
        perror(name);
    } else {
        try {
            copyToHandle(*arch.getFile(), mem.data_pos, mem.size, fd);
            close(fd);

            if(chmod(name.c_str(), mode) != 0)
                perror(name + ": chmod failed: ", LOG_WARN);
            if(preserve_flag) {
                struct utimbuf u;
                u.actime = u.modtime = parseNumber(mem.member.mtime, 10);
                if(utime(name.c_str(), &u) != 0)
                    perror(name + ": couldn't set timestamp: ", LOG_WARN);
            }
        }
        catch(...) {
            close(fd);
            std::remove(name.c_str());
            throw;
        }
    }
}

/*
 *  Archive Iteration
 *
 *  `time $AR p /usr/lib/libc.a vm86.o | wc'
 *    New Implementation:  0.080 real
 *    Old Implementation:  0.060 real
 *    GNU ar:              0.217 real
 *  `time $AR pv /usr/lib/libc.a | wc'
 *    New Implementation:  0.290 real
 *    Old Implementation:  0.250 real
 *    GNU ar:              0.440 real
 *  (iar compiled with g++-2.95.2 -O, GNU ar from vanilla SuSE 5.1;
 *  output is bit-for-bit identical; Pentium 200, idle)
 */

/*
 *  Iterate through an archive, calling the specified closure for each
 *  member, in order.
 *
 *  Though it's a bit slower than the old implementation, I like this
 *  better because it uses the /Archive/ class. The old one used
 *  /ArchiveReader/ (R.I.P.) which was more complicated.
 */
void iarIterateInOrder(const Callee2<Archive&, Archive::MemberData&, void>& what,
                       int filemode, bool armap)
{
    Ptr<File> f = new File(arch_name, filemode);
    if(!f->isOpen())
        throw LinkerError(string_t("Can't open `") + arch_name + "': " +
                          f->openErrorMessage());

    Archive arch(new VFile(f, 0, -1), armap);

    if(file_names.empty()) {
        /* show all members */
        Archive::Iterator it(arch);
        if(it.valid())
            do {
                what.call(arch, *it);
            } while(it.next());
    } else {
        /* show only members in /file_names/ */
        for(std::vector<const char*>::iterator fn_iter = file_names.begin();
            fn_iter != file_names.end(); ++fn_iter) {
            Archive::MPtr mptr = arch.getMemberByName(*fn_iter);
            if(mptr.ptr()) {
                what.call(arch, *mptr);
            } else {
                log(LOG_ERROR) << "no entry " << *fn_iter << " in archive\n";
            }
        }
    }
}
