/**
  *  \file got.cc
  */

#include "got.h"
#include "output.h"
#include "table.h"
#include "cpluslib/assert.h"
#include "symbol.h"
#include "state.h"

// Header used for GOT
static const Elf32_Shdr got_header = {
    0,                          // sh_name
    SHT_PROGBITS,               // sh_type
    SHF_ALLOC,                  // sh_flags
    0,                          // sh_addr
    0,                          // sh_offset
    0,                          // sh_size (will be modified)
    0,                          // sh_link
    0,                          // sh_info
    4,                          // sh_addralign -- FIXME?
    0                           // sh_entsize
};

GlobalOffsetTable::GlobalOffsetTable()
    : ProgramSection(0, got_header)
{
    setName(".got");
}

GlobalOffsetTable::~GlobalOffsetTable()
{ }

Elf32_Word
GlobalOffsetTable::addEntry(const string_t& name)
{
    Elf32_Word adr = header.sh_size;
    header.sh_size += sizeof(Elf32_Word);
    entries.push_back(name);
    return adr;
}

void
GlobalOffsetTable::writeOutput(Output& out)
{
    CTASSERT(sizeof(Elf32_Word) == 4);
    ASSERT(entries.size() * sizeof(Elf32_Word) == header.sh_size);
    ASSERT(header.sh_size);
    
    VLArray<Elf32_Addr> image(header.sh_size / sizeof(Elf32_Word));
    Elf32_Word index = 0;
    for(list_type::const_iterator i = entries.begin(); i != entries.end(); ++i) {
        image[index] = symbols.getAddressOf(*i);
        ++index;
    }
    out.writeOutput(image.getBuffer(),
                    address,
                    header.sh_size,
                    header.sh_size,
                    PF_R | PF_W);
}

void
GlobalOffsetTable::writeGotStateFile(StateFileWriter& sw)
{
    /* write global offset table */
    ElfTableWriter<Elf32_Sym> sym_writer(sw.getWriter(), SHT_SYMTAB, state_got_sec_name);
    ElfStringWriter           sym_str_writer(sw.getWriter(), state_got_str_name);
    sw.getWriter().setLink(sym_writer.getIndex(), sym_str_writer.getIndex());

    for (list_type::iterator i = entries.begin(); i != entries.end(); ++i) {
        /* FIXME: this is bogus. */
        Elf32_Sym s;
        s.st_name  = sym_str_writer.addString(*i);
        s.st_value = 0;
        s.st_size  = 0;             // not used in ild
        s.st_info  = 0;
        s.st_other = 0;
        s.st_shndx = SHN_ABS;
        sym_writer.add(s);
    }
}
