#include <cstring>
#include "getopt.h"
#include "cpluslib/string.h"
#include "cpluslib/assert.h"

// Identify a long option. Long options can be abbreviated to the
// shortest unique prefix. Throws exception when no option found.
static const Option&
findLongOption(const Option* array, const char* name)
{
    const Option* found = 0;             // points to first found entry
    std::size_t len = std::strlen(name); // length of option
    string_t ambig;                      // for reporting of ambiguous options
    
    while(array->short_name || array->long_name) {
        if(array->long_name) {
            if(std::strncmp(name, array->long_name, len) == 0) {
                /* identical match? */
                if(std::strlen(array->long_name) == len)
                    return *array;

                /* partial match */
                if(!found)
                    found = array;
                else
                    ambig = ambig + ", --" + array->long_name;
            }
        }
        ++array;
    }
    if(ambig.length())
        throw Option::Error(string_t("Ambiguous long option `--") + name
                            + "': can be --" + found->long_name + ambig);
    if(!found)
        throw Option::Error(string_t("Invalid option `--") + name + "'");

    return *found;
}

// Identify a short option. Throws if none found.
static const Option&
findShortOption(const Option* array, char name)
{
    while(array->short_name || array->long_name) {
        if(array->short_name == name)
            return *array;
        ++array;
    }
    throw Option::Error(string_t("Invalid option `-") + name + "'");
}

// Make an option iterator at the end of the option list.
Option::iterator::iterator()
    : defs(0), argv(0), argp(0), stopped(true)
{ }

// Make an option iterator at the beginning of the option list.
// /aargv/ is the argv passed to main. /adefs/ defines the options.
Option::iterator::iterator(char** aargv, const Option* adefs)
    : defs(adefs), argv(aargv+1), argp(0), stopped(false)
{
    next();
}

//
//  Advance to next option. Current status is, at any time:
//
//  * stopped = true: we encountered the option-stop character (`--'),
//    or user called `stop()'. argp is null then, argv points to next
//    argument or is null.
//  * argp non-null: we're parsing a string of single-character options.
//    argp points to the next option to return.
//  * argp null, argv non-null: we have to parse a new option next time,
//    argv points to it.
//  * argp and argv null: end of option list reached.
//  * argp never points to a null character, argv never points to a
//    null pointer.
//
void
Option::iterator::next()
{
    /* after `stop', return all arguments verbatim */
    if(stopped) {
        ASSERT(!argp);
        id = 0;
        arg = *argv++;
        if(!arg)
            argv = 0;
        return;
    }

    /* parsing a new option */
    if(!argp) {
        char* p = *argv++;
        if(!p) {
            /* end of command line reached */
            argv = 0;
            return;
        }

        if(*p != '-' || p[1] == 0) {
            /* non-option. `-' is a non-option for this purpose. */
            id = 0;
            arg = p;
            return;
        }

        if(p[1] == '-' && p[2] == 0) {
            /* stop signal */
            stop();
            next();
            return;
        }

        if(p[1] == '-') {
            /* long option */
            p += 2;
            char* eq = std::strchr(p, '=');
            if(eq)
                *eq = 0;
            const Option& opt = findLongOption(defs, p);
            id = opt.ret_val ? opt.ret_val : opt.short_name;
            if(opt.arg) {
                /* needs argument */
                if(eq) {
                    arg = eq+1;
                } else {
                    arg = *argv++;
                    if(!arg)
                        throw Error(string_t("Option needs an argument: `--") + p + "'");
                }
            } else {
                /* no argument */
                if(eq)
                    throw Error(string_t("Option `--") + p + "' does not allow an argument");
                arg = 0;
            }
            return;
        }

        /* okay, it is a (sequence of) short options */
        argp = p+1;
    }

    /* argp now nonzero, pointing to current character */
    ASSERT(argp);
    ASSERT(*argp);
    char optc = *argp++;
    if(!*argp)
        argp = 0;
    const Option& opt = findShortOption(defs, optc);
    if(opt.arg) {
        /* needs argument */
        if(argp) {
            arg = argp;
            argp = 0;
        } else {
            arg = *argv++;
            if(!arg)
                throw Option::Error(string_t("Option needs an argument: `-") + optc + "'");
        }
    } else
        arg = 0;
    id = opt.ret_val ? opt.ret_val : opt.short_name;
}
