/**
  *  \file file.cc
  *  \brief Stateless Unbuffered POSIX File I/O
  *
  *  This module provides a lightweight wrapper around POSIX file I/O.
  */
#define _BSD_SOURCE      // fchmod
#include <cstring>              // strerror
#include <cstddef>              // size_t
#include <cerrno>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "file.h"
#include "config.h"

/** \class File
    \brief Stateless Unbuffered POSIX File I/O

    This class provides stateless (no lseek) unbuffered (no stdio)
    file access using the POSIX system calls.

    We're moving around large binary blocks of which we know the
    size. Buffering doesn't get us anything here.

    We usually do random access. Having to lseek explicitly only
    clutters up the code. We assume lseek doesn't cost anything;
    but caching could be added easily.

    \see VFile
*/

/** Statistics on I/O. We count the bytes requested, not the bytes
    actually read: when our files are well-formed, there are no partial
    reads */
unsigned long File::bytes_read = 0, File::bytes_written = 0;

/** Open file. Wrapper for POSIX' ::open(). Use /isOpen()/ to check
    for errors */
File::File(const char* filename, int flags)
    : fd(-1)
{
    open(filename, flags);
}

/** Close file. */
File::~File()
{
    close();
}

/** Open file. Returns true if ok, false on error. */
bool
File::open(const char* filename, int flags)
{
    close();
    fd = ::open(filename, flags);
    open_errno = (fd >= 0 ? 0 : errno);
    return fd >= 0;
}

/** Create file. Returns true if ok, false on error. */
bool
File::create(const char* filename, int mode)
{
    close();
    fd = ::open(filename, O_RDWR | O_CREAT | O_TRUNC, mode);
    open_errno = (fd >= 0 ? 0 : errno);
    return fd >= 0;
}

/** Close file. */
void
File::close()
{
    if(fd >= 0)
        ::close(fd);
    fd = -1;
}

/** Read a block. Returns number of bytes read, or -1 on error. */
ssize_t
File::read(off_t where, void* buffer, size_t count)
{
    if(::lseek(fd, where, SEEK_SET) >= 0) {
        bytes_read += count;
        return ::read(fd, buffer, count);
    } else
        return 0;
}

/** Write a block. Returns number of bytes written. */
ssize_t
File::write(off_t where, const void* buffer, size_t count)
{
    if(::lseek(fd, where, SEEK_SET) >= 0) {
        bytes_written += count;
        return ::write(fd, buffer, count);
    } else
        return 0;
}

/** Get error message from last open. */
string_t
File::openErrorMessage()
{
    return std::strerror(open_errno);
}

/** Change file permissions. File must be open. */
int
File::chmod(mode_t mode)
{
    if(isOpen())
        return fchmod(fd, mode);
    else
        return -1;
}

/** Return file statistics (mode, inode, mtime,...) */
int
File::fstat(struct stat& buf)
{
    return ::fstat(fd, &buf);
}

/** Returns file modification time. */
time_t
File::getMtime()
{
    struct stat statbuf;
    if(fstat(statbuf) == 0)
        return statbuf.st_mtime;
    else
        return -1;
}

int
File::fixMode(int mode)
{
    int um = umask(0);
    umask(um);
    return mode & ~um;
}
