/**
  *  \file elfio.h
  */
#ifndef ILD_ELFIO_H
#define ILD_ELFIO_H

#include "vfile.h"
#include "cpluslib/smartptr.h"
#include "cpluslib/buffer.h"
#include "cpluslib/string.h"
#include "elf.h"
#include "cpluslib/assert.h"

class ElfWriter {
    Elf32_Ehdr          header;
    Buffer<Elf32_Shdr>  shdrs;
    string_t            sec_names;
    Ptr<VFile>          file;
    Elf32_Word          pos;
 public:
    ElfWriter(Ptr<VFile> f, Elf32_Half type);
    ~ElfWriter();
    void close();

    /* allocate a section slot */
    Elf32_Word  addSection(Elf32_Word type, string_t name);
    /* set section data */
    void setSectionData(Elf32_Word index, const void* data, Elf32_Word size);
    /* used to edit section header */
    Elf32_Shdr& getSection(Elf32_Word num) { return shdrs[num]; }
    /* linking two sections */
    void setLink(Elf32_Word sec, Elf32_Word target) { shdrs[sec].sh_link = target; }
};

class ElfStringWriter {
    ElfWriter& w;
    Elf32_Word index;
    string_t content;
 public:
    ElfStringWriter(ElfWriter& w, string_t name);
    ~ElfStringWriter();
    void write();
    Elf32_Word addString(const string_t& str);
    Elf32_Word getIndex() const { return index; }
};

template<class T>
class ElfTableWriter {
    ElfWriter& w;
    Elf32_Word index;
    Buffer<T> content;
 public:
    ElfTableWriter(ElfWriter& w, Elf32_Word typ, string_t name)
        : w(w), index(w.addSection(typ, name)), content() { ASSERT(index); }
    ~ElfTableWriter() { write(); }
    void write();
    Elf32_Word add(const T& t) { ASSERT(index); Elf32_Word rv = content.getSize(); content.append(t); return rv; }
    Elf32_Word getIndex() const { return index; }
};

template<class T>
void
ElfTableWriter<T>::write()
{
    if (!index)
        return;
    w.getSection(index).sh_entsize = sizeof(T);
    w.setSectionData(index, content.ptr(), content.getSize() * sizeof(T));
    index = 0;
}

void initElfHeader(Elf32_Ehdr& h);

#endif
