/**
  *  \file elfio.cc
  */

#include "elfio.h"
#include "cpluslib/assert.h"

ElfWriter::ElfWriter(Ptr<VFile> f, Elf32_Half type)
    : file(f)
{
    initElfHeader(header);
    header.e_type = type;
    pos = sizeof(header);
    addSection(SHT_NULL, "");
}

ElfWriter::~ElfWriter()
{
    close();
}

void
ElfWriter::close()
{
    if (!file)
        return;

    /* Write string table with section names. */
    Elf32_Word strsec = addSection(SHT_STRTAB, ".shstrtab");
    Elf32_Word strlen = sec_names.length();
    setSectionData(strsec, sec_names.data(), strlen);

    /* Write section header */
    file->write(pos, shdrs.ptr(), shdrs.getSize() * sizeof(Elf32_Shdr));

    /* Write ELF header */
    header.e_shoff    = pos;
    header.e_shnum    = shdrs.getSize();
    header.e_shstrndx = strsec;
    file->write(0, &header, sizeof(header));
    file = 0;
}

Elf32_Word
ElfWriter::addSection(Elf32_Word type, string_t name)
{
    Elf32_Shdr shdr;
    shdr.sh_name      = sec_names.length();
    shdr.sh_type      = type;
    shdr.sh_flags     = 0;
    shdr.sh_addr      = 0;
    shdr.sh_offset    = 0;
    shdr.sh_size      = 0;
    shdr.sh_link      = 0;
    shdr.sh_info      = 0;
    shdr.sh_addralign = 0;
    shdr.sh_entsize   = 0;

    sec_names.append(name);
    sec_names.append(1, char(0));
    shdrs.append(shdr);

    return shdrs.getSize() - 1;
}

void
ElfWriter::setSectionData(Elf32_Word index, const void* data, Elf32_Word size)
{
    shdrs[index].sh_size = size;
    if (data) {
        shdrs[index].sh_offset = pos;
        file->write(pos, data, size);
        pos += size;
    }
}

void
initElfHeader(Elf32_Ehdr& header)
{
    for(int i = 0; i < EI_NIDENT; ++i)
        header.e_ident[i] = 0;
    
    header.e_ident[0]          = 127;
    header.e_ident[1]          = 'E';
    header.e_ident[2]          = 'L';
    header.e_ident[3]          = 'F';
    header.e_ident[EI_CLASS]   = ELFCLASS32;
    header.e_ident[EI_DATA]    = ELFDATA2LSB;
    header.e_ident[EI_VERSION] = EV_CURRENT;

    header.e_type      = ET_EXEC;
    header.e_machine   = EM_386;
    header.e_version   = EV_CURRENT;
    header.e_entry     = 0;
    header.e_phoff     = 0;
    header.e_shoff     = 0;
    header.e_flags     = 0;
    header.e_ehsize    = sizeof(Elf32_Ehdr);
    header.e_phentsize = sizeof(Elf32_Phdr);
    header.e_phnum     = 0;
    header.e_shentsize = sizeof(Elf32_Shdr);
    header.e_shnum     = 0;
    header.e_shstrndx  = 0;
}

/**************************** ElfStringWriter ****************************/

ElfStringWriter::ElfStringWriter(ElfWriter& w, string_t name)
    : w(w), index(w.addSection(SHT_STRTAB, name)), content(1, char(0))
{
    ASSERT(index);
}

ElfStringWriter::~ElfStringWriter()
{
    write();
}

void
ElfStringWriter::write()
{
    if (!index)
        return;
    w.setSectionData(index, content.data(), content.length());
    index = 0;
}

Elf32_Word
ElfStringWriter::addString(const string_t& str)
{
    ASSERT(index);
    Elf32_Word rv = content.length();
    content.append(str);
    content.append(1, char(0));
    return rv;
}
