#include <iostream>
#include <cstdio>
#include <vector>
#include <fcntl.h>
#include "elf.h"
#include "file.h"
#include "namemap.h"
#include "names.inc"
#include "table.h"
#include "cpluslib/string.h"

using namespace std;

string_t cur_file;                // current file

std::vector<string_t> section_names;

void error(string_t err)
{
    cerr << cur_file << ": " << err << endl;
}

void show_elf_header(Elf32_Ehdr& header)
{
    cout << "File " << cur_file << ":" << endl
         << "+ Header:" << endl
         << "    e_type      = " << getName(et_names, header.e_type) << endl
         << "    e_machine   = " << getName(em_names, header.e_machine) << endl
         << "    e_version   = " << header.e_version << endl
         << "    e_entry     = " << itox(header.e_entry) << endl
         << "    e_phoff     = " << itox(header.e_phoff) << endl
         << "    e_shoff     = " << itox(header.e_shoff) << endl
         << "    e_flags     = " << itox(header.e_flags) /*getBitfield(ef_names, header.e_flags)*/ << endl
         << "    e_ehsize    = " << itox(header.e_ehsize) << endl
         << "    e_phentsize = " << itox(header.e_phentsize) << endl
         << "    e_phnum     = " << itox(header.e_phnum) << endl
         << "    e_shentsize = " << itox(header.e_shentsize) << endl
         << "    e_shnum     = " << itox(header.e_shnum) << endl
         << "    e_shstrndx  = " << itox(header.e_shstrndx) << endl;
}

void show_dynamic(File& file, Elf32_Phdr& h)
{
    size_t n = h.p_filesz / sizeof(Elf32_Dyn);
    VLArray<Elf32_Dyn> dyn(h.p_filesz);

    file.read(h.p_offset, dyn.getBuffer(), n * sizeof(Elf32_Dyn));
    for(size_t i = 0; i < n; i++) {
        string_t s = getName(dt_names, dyn[i].d_tag);
        while(s.length() < 16)
            s += " ";
        cout << "      " << s << " " << itox(dyn[i].d_un.d_val) << endl;
    }
}

void show_program_header(File& file, Elf32_Ehdr& header)
{
    if(header.e_phoff == 0) {
        cout << "+ Program header table: not present" << endl;
        return;
    }
    cout << "+ Program header table:" << endl;

    Table<Elf32_Phdr> phdr(header.e_phentsize, header.e_phnum);
    file.read(header.e_phoff, phdr.getBuffer(), phdr.getSize());

    for(size_t i = 0; i < header.e_phnum; i++) {
        Elf32_Phdr p = phdr[i];
        cout << "  - Entry #" << i << endl
             << "    p_type   = " << getName(pt_names, p.p_type) << endl
             << "    p_offset = " << itox(p.p_offset) << endl
             << "    p_vaddr  = " << itox(p.p_vaddr) << endl
             << "    p_filesz = " << itox(p.p_filesz) << endl
             << "    p_memsz  = " << itox(p.p_memsz) << endl
             << "    p_flags  = " << getBitfield(pf_names, p.p_flags) << endl
             << "    p_align  = " << itox(p.p_align) << endl;
        if(p.p_type == PT_DYNAMIC)
            show_dynamic(file, p);
    }
}

void show_shdr(Elf32_Shdr& h, VLArray<char>& strtab, const char* pfx)
{
    cout << pfx << "    sh_name      = " << h.sh_name << " (" << strtab.getBuffer() + h.sh_name << ")" << endl
         << pfx << "    sh_type      = " << getName(sht_names, h.sh_type) << endl
         << pfx << "    sh_flags     = " << getBitfield(shf_names, h.sh_flags) << endl
         << pfx << "    sh_addr      = " << itox(h.sh_addr) << endl
         << pfx << "    sh_offset    = " << itox(h.sh_offset) << endl
         << pfx << "    sh_size      = " << itox(h.sh_size) << endl
         << pfx << "    sh_link      = " << h.sh_link << endl
         << pfx << "    sh_info      = " << h.sh_info << endl
         << pfx << "    sh_addralign = " << itox(h.sh_addralign) << endl
         << pfx << "    sh_entsize   = " << itox(h.sh_entsize) << endl;
};

void show_file_table(File& file, Elf32_Shdr& shdr, Table<Elf32_Shdr>& tab)
{
    if (shdr.sh_link >= tab.getCount()) {
        error("invalid sh_link for file table");
        return;
    }
    Elf32_Shdr& strtab_shdr = tab[shdr.sh_link];
    VLArray<char> strtab(strtab_shdr.sh_size);
    file.read(strtab_shdr.sh_offset, strtab.getBuffer(), strtab_shdr.sh_size);

    size_t n = shdr.sh_size / shdr.sh_entsize;
    Table<Elf32_ild_FileEntry> fe(shdr.sh_entsize, n);
    file.read(shdr.sh_offset, fe.getBuffer(), fe.getSize());

    for (size_t i = 0; i < n; ++i) {
        Elf32_ild_FileEntry& e = fe[i];
        cout << "    - " << itox(i) << ": " << strtab.getBuffer() + e.ife_name
             << "\n      . ife_mtime   = " << e.ife_mtime
             << "\n      . ife_symbols = " << itox(e.ife_symbols)
             << "\n      . ife_sections = " << itox(e.ife_sections)
             << "\n";
    }
}

void show_symbol_table(File& file, Elf32_Shdr& shdr, Table<Elf32_Shdr>& tab)
{
    if(/*shdr.sh_link < 0 ||*/ shdr.sh_link >= tab.getCount()) {
        error("invalid symbol header links");
        return;
    }

    Elf32_Shdr& strtab_shdr = tab[shdr.sh_link];
    VLArray<char> strtab(strtab_shdr.sh_size);
    file.read(strtab_shdr.sh_offset, strtab.getBuffer(), strtab_shdr.sh_size);

    size_t n = shdr.sh_size / shdr.sh_entsize;
    Table<Elf32_Sym> sym(shdr.sh_entsize, n);
    file.read(shdr.sh_offset, sym.getBuffer(), sym.getSize());

    for(size_t i = 0; i < n; i++) {
        Elf32_Sym& s = sym[i];

        cout << "    " << itox(i) << ": " << strtab.getBuffer() + s.st_name << endl
             << "      " << getName(stb_names, ELF32_ST_BIND(s.st_info))
             << ", " << getName(stt_names, ELF32_ST_TYPE(s.st_info))
             << ", ";
        if (s.st_shndx && s.st_shndx < section_names.size())
            cout << section_names[s.st_shndx] << "/";
        cout << getName(shn_names, s.st_shndx) << ":" << itox(s.st_value)
             << ", size " << s.st_size << endl;
    }
}

void show_rel_table(File& file, Elf32_Shdr& shdr)
{
    size_t n = shdr.sh_size / shdr.sh_entsize;
    Table<Elf32_Rel> rel(shdr.sh_entsize, n);
    file.read(shdr.sh_offset, rel.getBuffer(), rel.getSize());

    for(size_t i = 0; i < n; i++) {
        Elf32_Rel& r = rel[i];
        cout << "      " << itox(r.r_offset) << " -> symbol "
             << itox(ELF32_R_SYM(r.r_info)) << ", "
             << getName(r_names, ELF32_R_TYPE(r.r_info)) << endl;
    }
}

void show_seclist(File& file, Elf32_Shdr& shdr, Table<Elf32_Shdr>& tab)
{
    if (shdr.sh_link >= tab.getCount()) {
        error("invalid sh_link for section list");
        return;
    }
    Elf32_Shdr& strtab_shdr = tab[shdr.sh_link];
    VLArray<char> strtab(strtab_shdr.sh_size);
    file.read(strtab_shdr.sh_offset, strtab.getBuffer(), strtab_shdr.sh_size);

    size_t n = shdr.sh_size / shdr.sh_entsize;
    Table<Elf32_Shdr> sh(shdr.sh_entsize, n);
    file.read(shdr.sh_offset, sh.getBuffer(), sh.getSize());

    for (size_t i = 0; i < n; ++i) {
        cout << "    - Section " << i << endl;
        show_shdr(sh[i], strtab, "  ");
    }
}
    
void show_section_header(File& file, Elf32_Ehdr& header)
{
    if(header.e_shoff == 0) {
        cout << "+ Section header table: not present" << endl;
        return;
    }
    cout << "+ Section header table:" << endl;

    Table<Elf32_Shdr> shdr(header.e_shentsize, header.e_shnum);
    file.read(header.e_shoff, shdr.getBuffer(), shdr.getSize());

    if(header.e_shstrndx >= shdr.getCount() || shdr[header.e_shstrndx].sh_type != SHT_STRTAB) {
        error("invalid string table index in ELF header");
        return;
    }

    Elf32_Shdr& strtab_shdr = shdr[header.e_shstrndx];
    VLArray<char> strtab(strtab_shdr.sh_size);
    file.read(strtab_shdr.sh_offset, strtab.getBuffer(), strtab_shdr.sh_size);

    for (size_t i = 0; i < shdr.getCount(); ++i)
        section_names.push_back(strtab.getBuffer() + shdr[i].sh_name);

    for(size_t i = 0; i < shdr.getCount(); i++) {
        Elf32_Shdr& h = shdr[i];
        cout << "  . Section " << i << endl;
        show_shdr(h, strtab, "");
        switch(h.sh_type) {
         case SHT_SYMTAB:
         case SHT_DYNSYM:
            show_symbol_table(file, h, shdr);
            break;
         case SHT_REL:
            show_rel_table(file, h);
            break;
         case SHT_ild_FILELIST:
            show_file_table(file, h, shdr);
            break;
         case SHT_ild_SECTIONS:
            show_seclist(file, h, shdr);
            break;
        }
    }
}

void dump_elf(const char* filename)
{
    File elf(filename, O_RDONLY);
    if(!elf.isOpen()) {
        std::perror(filename);
        return;
    }
    cur_file = filename;

    Elf32_Ehdr header;
    ssize_t n;
    if((n = elf.read(0, &header, sizeof(header))) != sizeof(header)
       || header.e_ident[0] != 127
       || header.e_ident[1] != 'E'
       || header.e_ident[2] != 'L'
       || header.e_ident[3] != 'F') {
        error("not an ELF file");
        return;
    }
    if(header.e_ident[EI_CLASS] != ELFCLASS32
       || header.e_ident[EI_DATA] != ELFDATA2LSB) {
        error("invalid ELF class");
        return;
    }

    /* ok, looks like we can grok that file */
    show_elf_header(header);
    show_section_header(elf, header);
    show_program_header(elf, header);
}

int main(int /*argc*/, char** argv)
{
    while(*++argv)
        dump_elf(*argv);
}
