/*! \file elf.h
 *
 *  ELF definitions.
 *
 *  This contains the ELF file format specifications \ild uses, and the
 *  definitions used for the state file.
 *
 *  Derived from <linux/elf.h>, the ELF Standard and some peeks into the
 *  GNU binutils headers. Differences to the Linux version:
 *  - Elf32/i386 only. Well, almost.
 *  - works with -ansi
 *  - more logical order ;)
 */
#ifndef ILD_ELF_H
#define ILD_ELF_H

/****************************** Base Types ******************************/

typedef unsigned long   Elf32_Addr;  //!< Machine address
typedef unsigned short  Elf32_Half;  //!< Half word
typedef unsigned long   Elf32_Off;   //!< Offset (added to Address)
typedef signed long     Elf32_Sword; //!< Signed machine word
typedef unsigned long   Elf32_Word;  //!< Machine word

/**************************** ELF file header ***************************/

#define EI_NIDENT       16           //!< Size of machine-independant part of Elf32_Ehdr

/**
  *  ELF file header. This header is used for 32-bit object and executable
  *  files. ild currently only reads "native-format" ELF, so this header is
  *  read in binary from disk.
  */
struct Elf32_Ehdr {
    unsigned char e_ident[EI_NIDENT];
    Elf32_Half    e_type;         // ET_xxx
    Elf32_Half    e_machine;      // EM_xxx
    Elf32_Word    e_version;      // EV_xxx
    Elf32_Addr    e_entry;        // entry point (virtual address)
    Elf32_Off     e_phoff;        // -> Elf32_Phdr[e_phnum]
    Elf32_Off     e_shoff;        // -> Elf32_Shdr[e_shnum]
    Elf32_Word    e_flags;        // EF_xxx (none defined for Intel)
    Elf32_Half    e_ehsize;       // sizeof(Elf32_Ehdr)
    Elf32_Half    e_phentsize;    // sizeof(Elf32_Phdr)
    Elf32_Half    e_phnum;
    Elf32_Half    e_shentsize;    // sizeof(Elf32_Shdr)
    Elf32_Half    e_shnum;
    Elf32_Half    e_shstrndx;
};

//@{
//! Elf32_Ehdr::e_ident indices
#define EI_MAG0         0
#define EI_MAG1         1
#define EI_MAG2         2
#define EI_MAG3         3
#define EI_CLASS        4
#define EI_DATA         5
#define EI_VERSION      6
#define EI_PAD          7
//@}

//@{
//! ELF magic numbers
#define ELFMAG0         0x7f
#define ELFMAG1         'E'
#define ELFMAG2         'L'
#define ELFMAG3         'F'
#define ELFMAG          "\177ELF"
#define SELFMAG         4
//@}

//@{
//! ELF classes (Elf32_Ehdr::e_ident[EI_CLASS])
#define ELFCLASSNONE    0
#define ELFCLASS32      1       // we use this one...
#define ELFCLASS64      2
#define ELFCLASSNUM     3
//@}

//@{
//! ELF byte ordering (Elf32_Ehdr::e_ident[EI_DATA])
#define ELFDATANONE     0
#define ELFDATA2LSB     1
#define ELFDATA2MSB     2
//@}

//@{
//! ELF version (Elf32_Ehdr::e_version)
#define EV_NONE         0
#define EV_CURRENT      1
#define EV_NUM          2
//@}

//@{
//! ELF file types (Elf32_Ehdr::e_type).
#define ET_NONE         0
#define ET_REL          1       ///< .o
#define ET_EXEC         2       ///< .exe
#define ET_DYN          3       ///< .so
#define ET_CORE         4
#define ET_ild_STATE    0x7473  ///< 'st' -- state files. Non-standard.
#define ET_LOPROC       0xFF00
#define ET_HIPROC       0xFFFF
//@}

//@{
//! ELF machine types (Elf32_Ehdr::e_machine)
#define EM_NONE         0
#define EM_M32          1       // AT&T WE 32100
#define EM_SPARC        2       // Sun SPARC
#define EM_386          3       // i386
#define EM_68K          4       // Motorola 68000
#define EM_88K          5       // Motorola 88000
#define EM_486          6       // (i486 -- not used in Intel's document)
#define EM_860          7       // Intel 80860
#define EM_MIPS         8       // MIPS R3000 (officially, big-endian only)
#define EM_MIPS_RS4_BE 10       // MIPS R4000 big-endian
#define EM_SPARC64     11       // SPARC v9 (not official) 64-bit
#define EM_PARISC      15       // HPPA
#define EM_SPARC32PLUS 18       // Sun's "v8plus"
#define EM_PPC         20       // PowerPC
#define EM_ALPHA       0x9026   // (Linux' interim value)
//@}

/***************************** Section Table ****************************/

/** Section header. */
struct Elf32_Shdr {
    Elf32_Word    sh_name;      ///< pointer into e_shstrndx
    Elf32_Word    sh_type;      ///< type, SHT_xxx
    Elf32_Word    sh_flags;     ///< flags, SHF_xxx
    Elf32_Addr    sh_addr;      ///< memory address
    Elf32_Off     sh_offset;    ///< file position. In state file: relocation address
    Elf32_Word    sh_size;      ///< size
    Elf32_Word    sh_link;      ///< -> other section
    Elf32_Word    sh_info;
    Elf32_Word    sh_addralign;
    Elf32_Word    sh_entsize;
};

//@{
//! Section types (Elf32_Shdr::sh_type)
#define SHT_NULL            0
#define SHT_PROGBITS        1       ///< program bits (data, text)
#define SHT_SYMTAB          2       ///< symbol table (Elf32_Sym[], see below)
#define SHT_STRTAB          3       ///< string table
#define SHT_RELA            4       ///< relocation (Elf32_Rela[], see below)
#define SHT_HASH            5       ///< hash table
#define SHT_DYNAMIC         6       ///< dynamic (Elf32_Dyn[], see below)
#define SHT_NOTE            7       ///< notes
#define SHT_NOBITS          8       ///< program bits (bss)
#define SHT_REL             9       ///< relocation (Elf32_Rel[], see below)
#define SHT_SHLIB           10
#define SHT_DYNSYM          11      ///< symbol table for dynamic loader
#define SHT_NUM             12
#define SHT_ild_FILELIST    0x646C6931 ///< Elf32_ild_FileEntry
#define SHT_ild_SECTIONS    0x646C6932 ///< Elf32_Shdr[]
#define SHT_SUNW_verdef     0x6FFFFFFD
#define SHT_SUNW_verneed    0x6FFFFFFE
#define SHT_SUNW_versym     0x6FFFFFFF
#define SHT_LOPROC          0x70000000
/* These are not used, but we document them here to avoid collisions. */
/* Alpha */
#  define SHT_ALPHA_DEBUG       0x70000001
#  define SHT_ALPHA_REGINFO     0x70000002
/* HP PA-RISC */
#  define SHT_PARISC_GOT        0x70000000
#  define SHT_PARISC_ARCH       0x70000001
#  define SHT_PARISC_GLOBAL     0x70000002
#  define SHT_PARISC_MILLI      0x70000003
#  define SHT_PARISC_UNWIND     0x70000004
#  define SHT_PARISC_PLT        0x70000005
#  define SHT_PARISC_SDATA      0x70000006
#  define SHT_PARISC_SBSS       0x70000007
#  define SHT_PARISC_SYMEXTN    0x70000008
#  define SHT_PARISC_STUBS      0x70000009
/* MIPS */
#  define SHT_MIPS_LIBLIST      0x70000000
#  define SHT_MIPS_MSYM         0x70000001
#  define SHT_MIPS_CONFLICT     0x70000002
#  define SHT_MIPS_GPTAB        0x70000003
#  define SHT_MIPS_UCODE        0x70000004
#  define SHT_MIPS_DEBUG        0x70000005
#  define SHT_MIPS_REGINFO      0x70000006
#  define SHT_MIPS_IFACE        0x7000000B
#  define SHT_MIPS_CONTENT      0x7000000C
#  define SHT_MIPS_OPTIONS      0x7000000D
#  define SHT_MIPS_DWARF        0x7000001E
#  define SHT_MIPS_SYMBOL_LIB   0x70000020
#  define SHT_MIPS_EVENTS       0x70000021
#define SHT_HIPROC          0x7fffffff
#define SHT_LOUSER          0x80000000
#define SHT_HIUSER          0xffffffff
//@}

//@{
//! Section flags (Elf32_Shdr::sh_flags)
#define SHF_WRITE       0x1
#define SHF_ALLOC       0x2
#define SHF_EXECINSTR   0x4
#define SHF_MASKPROC    0xf0000000
//@}

//@{
//! Special section numbers
#define SHN_UNDEF       0       // undefined symbols
#define SHN_LORESERVE   0xff00
#define SHN_LOPROC      0xff00
#define SHN_HIPROC      0xff1f
#define SHN_ABS         0xfff1  // symbols with absolute values
#define SHN_COMMON      0xfff2  // "common" blocks
#define SHN_HIRESERVE   0xffff
//@}

/***************************** Program Table ****************************/

/** Program header. Defines load image of a program. */
struct Elf32_Phdr {
    Elf32_Word  p_type;         ///< PT_xxx
    Elf32_Off   p_offset;       ///< position in file
    Elf32_Addr  p_vaddr;        ///< virtual address
    Elf32_Addr  p_paddr;        ///< physical address (unused in SysV)
    Elf32_Word  p_filesz;       ///< size in file
    Elf32_Word  p_memsz;        ///< size in memory
    Elf32_Word  p_flags;        ///< PF_xxx
    Elf32_Word  p_align;        ///< alignment constraints (power of 2)
};

//@{
//! Program table entries (Elf32_Phdr::p_type)
#define PT_NULL    0
#define PT_LOAD    1            // "load this into memory"
#define PT_DYNAMIC 2            // -> Elf32_Dyn[]
#define PT_INTERP  3            // -> "/lib/ld.so"
#define PT_NOTE    4            // notes
#define PT_SHLIB   5            // (unspecified semantics -- do not use)
#define PT_PHDR    6            // -> Elf32_Phdr[] (self-reference)
#define PT_LOPROC  0x70000000
#define PT_HIPROC  0x7fffffff
//@}

//@{
//! Permissions for a loadable section (Elf32_Phdr::p_flags)
#define PF_R            0x4
#define PF_W            0x2
#define PF_X            0x1
//@}

/**************************** Dynamic Section ***************************/

/** Dynamic section entry. */
struct Elf32_Dyn {
    Elf32_Sword d_tag;          // DT_xxx
    union {
        Elf32_Sword     d_val;
        Elf32_Addr      d_ptr;
    } d_un;
};

//@{
//! Dynamic section types (Elf32_Dyn::d_tag)
#define DT_NULL         0       ///< Flag: terminator for _DYNAMIC
#define DT_NEEDED       1       ///< d_val -> needed library name (in string table)
#define DT_PLTRELSZ     2       ///< d_val = size of relocation for PLT
#define DT_PLTGOT       3       ///< d_ptr -> _GLOBAL_OFFSET_TABLE_
#define DT_HASH         4       ///< d_ptr -> hash table
#define DT_STRTAB       5       ///< d_ptr -> string table
#define DT_SYMTAB       6       ///< d_ptr -> symbol table
#define DT_RELA         7       ///< d_ptr -> Elf32_Rela[]
#define DT_RELASZ       8       ///< d_val = size of the above
#define DT_RELAENT      9       ///< d_val = size of Elf32_Rela (?)
#define DT_STRSZ        10      ///< d_val = size of string table
#define DT_SYMENT       11      ///< d_val = size of Elf32_Sym
#define DT_INIT         12      ///< d_ptr -> initialisation function
#define DT_FINI         13      ///< d_ptr -> finalisation function
#define DT_SONAME       14      ///< d_val -> our lib name
#define DT_RPATH        15      ///< d_val -> search path
#define DT_SYMBOLIC     16      ///< Flag: modify symbol lookup
#define DT_REL          17      ///< d_ptr -> Elf32_Rel[]
#define DT_RELSZ        18      ///< d_val -> size of the above
#define DT_RELENT       19      ///< d_val -> size of Elf32_Rel
#define DT_PLTREL       20      ///< d_val = DT_REL / DT_RELA
#define DT_DEBUG        21
#define DT_TEXTREL      22      ///< Flag: relocation modifies .text/.rodata
#define DT_JMPREL       23      ///< d_ptr -> Elf32_Rel[]/Elf32_Rela[]
#define DT_LOPROC       0x70000000
#define DT_HIPROC       0x7fffffff
//@}

/***************************** Symbol Table *****************************/

/** Symbol table entry. */
struct Elf32_Sym {
    Elf32_Word    st_name;      ///< points to string table
    Elf32_Addr    st_value;     ///< symbol value (relative)
    Elf32_Word    st_size;
    unsigned char st_info;      ///< binding, type.
    unsigned char st_other;     ///< unused, zero
    Elf32_Half    st_shndx;
};

//@{
//! Symbol table entries: ELF32_ST_BIND(Elf32_Sym::st_info)
#define STB_LOCAL  0            // not externally visible
#define STB_GLOBAL 1            // globally visible
#define STB_WEAK   2            // used only if no global symbol
//@}

//@{
// Symbol table entries: ELF32_ST_TYPE(Elf32_Sym::st_info)
#define STT_NOTYPE  0
#define STT_OBJECT  1           // data object
#define STT_FUNC    2           // function
#define STT_SECTION 3           // STB_LOCAL, used for relocation
#define STT_FILE    4           // STB_LOCAL, name = file name
//@}

#define ELF32_ST_BIND(x)   ((x) >> 4)
#define ELF32_ST_TYPE(x)   (((unsigned int) x) & 0xf)
#define ELF32_ST_INFO(b,t) (((b) << 4) | ((t) & 0xf))

/*************************** Relocation Table ***************************/

/// Relocation entry with implicit addend.
struct Elf32_Rel {
    Elf32_Addr  r_offset;
    Elf32_Word  r_info;
};

/// Relocation entry with explicit addend.
struct Elf32_Rela {
    Elf32_Addr  r_offset;
    Elf32_Word  r_info;
    Elf32_Sword r_addend;
};

// Relocation entries
#define ELF32_R_SYM(x) ((x) >> 8)
#define ELF32_R_TYPE(x) ((x) & 0xff)

//@{
//! i386 relocation types. ELF32_R_TYPE(Elf32_Rela::r_info)
#define R_386_NONE      0
#define R_386_32        1       // S + A (absolute address)
#define R_386_PC32      2       // S + A - P (PC-relative)
#define R_386_GOT32     3       // G + A - P (offset into GOT)
#define R_386_PLT32     4       // L + A - P (offset into PLT)
#define R_386_COPY      5
#define R_386_GLOB_DAT  6       // ?
#define R_386_JMP_SLOT  7       // (?)
#define R_386_RELATIVE  8       // B + A (.so base address)
#define R_386_GOTOFF    9       // S + A - GOT (offset to GOT)
#define R_386_GOTPC     10      // GOT + A - P (PC-relative GOT address)
#define R_386_NUM       11
//@}
 
/******************************** Notes *********************************/

/// Notes.
struct Elf32_Nhdr {
    Elf32_Word    n_namesz;     // Name size
    Elf32_Word    n_descsz;     // Content size
    Elf32_Word    n_type;       // Content type
};

// notes used in cores
#define NT_PRSTATUS     1
#define NT_PRFPREG      2
#define NT_PRPSINFO     3
#define NT_TASKSTRUCT   4

/// These are used in core files, I think.
#define AT_NULL   0     /* end of vector */
#define AT_IGNORE 1     /* entry should be ignored */
#define AT_EXECFD 2     /* file descriptor of program */
#define AT_PHDR   3     /* program headers for program */
#define AT_PHENT  4     /* size of program header entry */
#define AT_PHNUM  5     /* number of program headers */
#define AT_PAGESZ 6     /* system page size */
#define AT_BASE   7     /* base address of interpreter */
#define AT_FLAGS  8     /* flags */
#define AT_ENTRY  9     /* entry point of program */
#define AT_NOTELF 10    /* program is not ELF */
#define AT_UID    11    /* real uid */
#define AT_EUID   12    /* effective uid */
#define AT_GID    13    /* real gid */
#define AT_EGID   14    /* effective gid */
#define AT_PLATFORM 15  /* string identifying cpu for optimizations */
#define AT_HWCAP  16    /* arch dependent hints at cpu capabilities */

/**************************** ild State Files ****************************/

/** Entry of `.ild.files' section (type SHT_ild_FILELIST) */
struct Elf32_ild_FileEntry {
    Elf32_Word    ife_name;     ///< file name
    Elf32_Word    ife_mtime;    ///< mtime
    Elf32_Word    ife_symbols;  ///< number of relevant symbol section
    Elf32_Word    ife_sections; ///< section contains section headers
};

#endif
