/**
  *  \file archive1.cc
  */

#include <cstdio>
#include "archive.h"
#include "log.h"
#include "cpluslib/types.h"
#include "vfile.h"
#include "parse.h"
#include "cpluslib/assert.h"

static inline uint32
roundUp(uint32 a)
{
    return a + (a & 1);
}

/*
 *  Move member /mem/ so that it begins after /min/.
 */
void
Archive::moveMember(MPtr mem, uint32 min)
{
    ASSERT(mem->pos < min);
    MPtr newmem = allocateSpace(mem->size, min);

    log(LOG_DEBUG) << "moving `" << mem->member.getFileName(lfntab)
                   << "' from " << std::hex << mem->pos << " to "
                   << newmem->pos << std::dec << "\n";

    ASSERT(newmem->pos >= min);
    ASSERT(mem->pos + sizeof(ArchiveMember) == mem->data_pos);

    /* copy member data */
    copyFileRegion(*file, mem->data_pos, newmem->data_pos, mem->size);
    changeArmapEntry(mem->pos, newmem->pos);

    /* change by_offset entries */
    by_offset.erase(mem->pos);
    by_offset[newmem->pos] = mem;
    mem->pos = newmem->pos;
    mem->data_pos = newmem->data_pos;
    mem->dirty = true;

    selfcheck();
}

/*
 *  Make sure that /needed/ bytes of headers can be written
 */
void
Archive::makeSpaceForHeaders(uint32 needed)
{
    off_t starter;
    off_t min = needed;

    // FIXME: shouldn't be necessary, but following code does not
    // cope with partially-read archives
    while(getNextMember())
        ;

    /* delete beginning of headers, find first data member */
    std::map<off_t, MPtr>::iterator i = by_offset.begin();
    while(i != by_offset.end() && i->second->type != M_FILE)
        by_offset.erase(i++);

    if(i == by_offset.end()) {
        starter = pos;          // archive is empty
        goto make_pad;          // FIXME: temporary hack
    }

    /* four cases:
       (1) exact match
       (2) too large. we have to move members out of the way.
       (3) too small. insert padding.
         (3a) enough space to insert a dummy member
         (3b) too small for an ArchiveMember header. move members
              out of the way */
    if(min != i->first)
        min += sizeof(ArchiveMember);

    log(LOG_DEBUG) << "header size is " << min << " (needed "
                   << needed << "), first member at " << i->first
                   << "\n";

    while(i != by_offset.end() && min > i->first) {
        /* a member must be moved (2 or 3b) */
        std::map<off_t, MPtr>::iterator t = i++;
        // FIXME: this fails if the archive has only one member
        moveMember(t->second, min);
    }

    /* okay, we made some space */
    starter = i->first;
 make_pad:
    if(needed < starter) {
        ASSERT(starter - needed >= sizeof(ArchiveMember));
        /* create padding */
        MPtr pad = createEmptyMember(needed, starter - needed - sizeof(ArchiveMember));
        pad->type = M_PAD;
        by_offset[needed] = pad;// FIXME: do this in createEmptyMember?
    }
}

void
Archive::saveDirtyMembers()
{
    static char zeroes[16];
    for(std::map<off_t, MPtr>::iterator i = by_offset.begin();
        i != by_offset.end(); ++i) {
        if(i->second->dirty) {
            storeNumber(i->second->member.size, 10, i->second->size);
            if(i->second->type == M_PAD) {
                char data[40];
                // FIXME: slash only on SysV
                std::sprintf(data, "$PAD.%ld/", (long)i->second->pos);
                storeString(i->second->member.filename, data);
            }
            file->write(i->second->pos, &i->second->member, sizeof(ArchiveMember));
            if(i->second->type == M_PAD && i->second->size > 0) {
                /* kill magic number to prevent an unaware program
                   from trying to parse this member */
                file->write(i->second->data_pos, zeroes,
                            std::min(i->second->size, uint32(sizeof(zeroes))));
            }
        }
    }
}

void
Archive::saveHeaders()
{
    /* ensure headers are read first... */
    while(where < W_FILES && getNextMember())
        ;

    /* the headers must have a fixed position. This may require
       relocation of members at the beginning of the file */

    /* first, compute new headers/positions */
    VLArray<char> armap(0);
    log(LOG_DEBUG) << "+ building armap...\n";
    buildArmap(armap);

    log(LOG_DEBUG) << "+ computing offsets...\n";
    const uint32 armap_ofs    = AR_MAGIC_LEN;
    const uint32 armap_dpos   = armap.getSize() == 0 ? armap_ofs : armap_ofs + sizeof(ArchiveMember);
    const uint32 lfntab_ofs   = armap_dpos + roundUp(armap.getSize());
    const uint32 lfntab_dpos  = lfntab.getSize() == 0 ? lfntab_ofs : lfntab_ofs + sizeof(ArchiveMember);
    const uint32 first_member = lfntab_dpos + roundUp(lfntab.getSize());

    log(LOG_DEBUG) << "+ making room for headers...\n";
    makeSpaceForHeaders(first_member);

    if(!armap_frob.empty()) {
        /* need to rebuild armap */
        log(LOG_DEBUG) << "+ updating armap again...\n";
        off_t old_size = armap.getSize();
        updateArmap();
        armap.changeSize(0);
        buildArmap(armap);
        ASSERT(armap.getSize() == old_size);
    }

    /* write new headers */
    log(LOG_DEBUG) << "+ maybe-writing armap...\n";
    if(armap.getSize()) {
        if(!armap_ptr.ptr()) {
            armap_ptr = createEmptyMember(armap_ofs, armap.getSize());
            storeString(armap_ptr->member.filename, "/");
        }
        armap_ptr->pos      = armap_ofs;
        armap_ptr->data_pos = armap_dpos;
        armap_ptr->size     = armap.getSize();
        armap_ptr->dirty    = true;
        armap_ptr->type     = M_ARMAP;
        storeNumber(armap_ptr->member.mtime, 10, armap_mtime);
        by_offset[armap_ofs] = armap_ptr;
        file->write(armap_dpos, armap.getBuffer(), armap.getSize());
    }
    log(LOG_DEBUG) << "+ maybe-writing lfntab...\n";
    if(lfntab.getSize()) {
        //ASSERT(lfntab_ptr.ptr());       // FIXME: may need to create
        if(!lfntab_ptr.ptr()) {
            lfntab_ptr = createEmptyMember(lfntab_ofs, lfntab.getSize());
            storeString(lfntab_ptr->member.filename, "//");
        }
        lfntab_ptr->pos      = lfntab_ofs;
        lfntab_ptr->data_pos = lfntab_dpos;
        lfntab_ptr->size     = lfntab.getSize();
        lfntab_ptr->dirty    = true;
        by_offset[lfntab_ofs] = lfntab_ptr;
        file->write(lfntab_dpos, lfntab.getBuffer(), lfntab.getSize());
    }

    log(LOG_DEBUG) << "+ selfcheck...\n";
    selfcheck();
    log(LOG_DEBUG) << "+ saving dirty headers...\n";
    saveDirtyMembers();
}
