#ifndef ILD_ARCHIVE_H
#define ILD_ARCHIVE_H

#include <map>
#include <sys/types.h>
#include "table.h"
#include "cpluslib/string.h"
#include "cpluslib/buffer.h"
#include "cpluslib/smartptr.h"

#define AR_MAGIC "!<arch>\012"
#define AR_MAGIC_LEN 8

class VFile;

/** Layout of an archive member header.
  * This must be a POD type. */
struct ArchiveMember {
    char filename[16];          ///< space-padded file name
    char mtime[12];             ///< seconds since epoch, decimal, space padded
    char uid[6];                ///< owner uid, decimal, space padded
    char gid[6];                ///< owner gid, decimal, space padded
    char mode[8];               ///< mode, octal, space padded
    char size[10];              ///< size, decimal, space padded
    char magic[2];              ///< "`\012"

    string_t getFileName(const VLArray<char>& names) const;
    off_t getSize() const;
    bool isNamed(const char* what) const;
};

/* Archive. */
class Archive {
 public:
    /** Type of a member of the archive */
    enum MemberType {
        M_FILE,                 ///< regular member
        M_ARMAP,                ///< armap
        M_ARMAPX,               ///< additional armap (MSVC++)
        M_LFNTAB,               ///< LFN table
        M_PAD                   ///< padding (dummy file)
    };
    /** Information about a member of the archive. */
    struct MemberData {
        off_t pos;              ///< position of member header in archive
        off_t data_pos;         ///< position of member in archive
        uint32 size;            ///< size of member
        MemberType type;        ///< type of member
        ArchiveMember member;   ///< member header
        bool dirty;             ///< changed?
    };
    typedef Ptr<MemberData> MPtr;
    class Iterator;
    friend class Iterator;
 private:
    /** Logical position inside the archive. */
    enum Where {
        W_START,                ///< we have not yet done anything
        W_ARMAP,                ///< we're expecting the armap
        W_ARMAPX,               ///< we're after the armap
        W_LFNTAB,               ///< we're expecting the LFN table
        W_FILES,                ///< we're expecting regular archive members
        W_EOF                   ///< we've seen EOF
    };
    Where                       where;
    Ptr<VFile>                  file;
    std::map<off_t, MPtr>       by_offset;
    std::map<string_t, MPtr>    by_name;
    VLArray<char>               lfntab;
    std::map<string_t, off_t>   armap;
    std::map<off_t, off_t>      armap_frob;
    off_t                       pos;
    bool                        parse_armap;
    MPtr                        armap_ptr, lfntab_ptr;
    uint32                      armap_mtime;
 public:
    Archive(Ptr<VFile> afile, bool aparse);
    Archive(Ptr<VFile> afile, int);

    MPtr  getMemberByName(string_t);
    MPtr  getMemberByOffset(off_t);
    off_t getOffsetOfSymbol(string_t);
    void  compactDirectory();
    void  updateArmap();
    MPtr  allocateSpace(uint32 size, uint32 min = 0);
    void  selfcheck();

    void  saveHeaders();
    void  saveDirtyMembers();

    void  clearArmap();
    void  changeArmapEntry(off_t old, off_t nw);
    void  addArmapEntry(string_t name, off_t offs);
    void  maybeBumpMtime(uint32 mtime) { if(mtime > armap_mtime) armap_mtime = mtime; }
    
    Ptr<VFile> getFile() const;
    const VLArray<char>& getLfnTab() const { return lfntab; }
    void  setLfnTab(const char* buf, std::size_t len);
 private:
    bool  getNextMember();
    void  parseBsdArmap(MemberData&);
    void  parseCoffArmap(MemberData&);
    void  buildArmap(VLArray<char>& array);
    void  buildCoffArmap(VLArray<char>& array);
    void  makeSpaceForHeaders(uint32 needed);
    void  moveMember(MPtr mem, uint32 min);
    MPtr  createEmptyMember(off_t pos, off_t size);
};

class Archive::Iterator {
    Archive& arch;
    std::map<off_t, MPtr>::iterator iter;
 public:
    Iterator(Archive& aarch);
    bool valid() const { return iter != arch.by_offset.end(); }
    bool next();
    MemberData& operator*() const { return *iter->second; }
    MemberData* operator->() const { return iter->second.ptr(); }
    Archive::MPtr ptr() const { return iter->second; }
};

#endif
