/**
  *  \file ar-symtab.cc
  */

#include <iostream>
#include "ar-symtab.h"
#include "elf.h"
#include "section.h"

void maybeReadSymtab(VFile& in, Archive& arch, Archive::MPtr new_mem,
                     const char* name)
{
    /* file opened, check format */
    Elf32_Ehdr  header;
    ssize_t n = in.read(0, &header, sizeof(header));
    if(n != sizeof(header) || header.e_ident[0] != 127
       || header.e_ident[1] != 'E'
       || header.e_ident[2] != 'L'
       || header.e_ident[3] != 'F')
    {
        std::cerr << name << ": not an ELF file\n";
        return;
    }
    
    if(header.e_ident[EI_CLASS] != ELFCLASS32
       || header.e_ident[EI_DATA] != ELFDATA2LSB
       || (header.e_type != ET_DYN && header.e_type != ET_REL)
       || header.e_shoff == 0)
    {
        std::cerr << name << ": invalid ELF file type\n";
        return;
    }
    
    /* read section headers */
    Table<Elf32_Shdr> shdr(header.e_shentsize, header.e_shnum);
    in.read(header.e_shoff, shdr.getBuffer(), shdr.getSize());

    size_t symtab_ndx = ~0U;
    for(size_t i = 0; i < shdr.getCount(); i++)
        if(shdr[i].sh_type == SHT_SYMTAB) {
            if(symtab_ndx != std::size_t(~0U))
                std::cerr << name << ": multiple symbol tables\n";
            symtab_ndx = i;
        }
    if(symtab_ndx == std::size_t(~0U)) {
        std::cerr << name << ": no symbol table\n";
        return;
    }

    /* ok, we have the symbol table index. Get the string table. */
    uint32 strtab_ndx = shdr[symtab_ndx].sh_link;
    if(strtab_ndx < 0 || strtab_ndx >= shdr.getCount()
       || shdr[strtab_ndx].sh_type != SHT_STRTAB) {
        std::cerr << name << ": invalid string table offset\n";
        return;
    }

    Ptr<StringTableSection> strtab = new StringTableSection(0, shdr[strtab_ndx]);
    strtab->loadFrom(in);

    /* read symbols */
    Elf32_Shdr& h = shdr[symtab_ndx];
    Table<Elf32_Sym> sym(h.sh_entsize, h.sh_size / h.sh_entsize);
    
    in.read(h.sh_offset, sym.getBuffer(), sym.getSize());
    for(std::size_t i = h.sh_info; i < sym.getCount(); ++i)
        if((ELF32_ST_BIND(sym[i].st_info) == STB_GLOBAL
            || ELF32_ST_BIND(sym[i].st_info) == STB_WEAK)
           && sym[i].st_shndx != SHN_UNDEF)
            arch.addArmapEntry(strtab->getString(sym[i].st_name), new_mem->pos);
}
