/*
 *  Simple String Testsuite
 *
 *  More-or-less random code that exercises all of the string
 *  members.
 *
 *  Compile with -DUSE_BUILTIN_STRING to test std::string instead of
 *  simple_string. The `compare' methods in g++'s std::string are
 *  fubar, hence they are not tested when detected.
 *
 *  This program should not output more than
 *      Testing <classname>...
 *      Done.
 *
 *  Call with numeric argument to repeat tests N times (for benchmarking).
 *  As of 03/Feb/2002, g++-2.95.2, P200, average of 3 x 1000 repetitions
 *    std::string       0.320 user
 *    simple_string     0.410 user    (original version)
 *    simple_string     0.415 user    (with capacity support)
 *  I'm unsure why it's slower with capacity support; appending ought
 *  to be much faster. OTOH, memory alignment seems to play much of a role
 *  on a stock Pentium (on a different program, I got +-10% by adding
 *  or removing a static variable). And for code size, this one wins
 *  big:
 *    std::string       46312 code + 3488 data
 *    simple_string     31518 code + 5712 data (with capacity support)
 */
#include "cpluslib/sstring.h"

#include <cstdlib>
#include <iostream>

/* Change to #if 0 or #if 1 to test simple_string or
   std::string, respectively. */
#ifdef USE_BUILTIN_STRING
#  include <string>
#  define simple_string std::string
#  define BUILTIN_STRING
#endif
#if defined(BUILTIN_STRING) && defined(__BASTRING__)
#  define BOGUS_STRING
#endif

bool
failed(const char* txt)
{
    std::cerr << "test failed: " << txt << "\n";
    return true;
}

bool
assert_fail(const char* expr, const char* file, int line)
{
    std::cerr << "assertion failed: " << expr << "\n"
              << "in " << file << ":" << line << "\n";
    std::exit(1);
}

#define TEST(expr) (void)((expr) || failed(#expr))

/* test ctors, assignment operator, assign() */
static void
test_init()
{
//     simple_string();
//     simple_string(const charT* src);
//     simple_string(const proxy&);
//     simple_string(const simple_string& rhs);
//     simple_string(const simple_string& str, size_type pos, size_type n = npos);
//     simple_string(const charT* src, size_type n);
//     simple_string(size_type n, charT c);

//     template<class InputIterator>
//     simple_string(InputIterator begin, InputIterator end);
//     simple_string& operator=(const simple_string& rhs);
//     simple_string& operator=(const charT* rhs);
//     simple_string& operator=(charT rhs);
//     simple_string& operator=(const proxy& rhs);
//     simple_string& assign(const simple_string& str);
//     simple_string& assign(const simple_string& str, size_type pos,
//                           size_type n);
//     simple_string& assign(const charT* s, size_type n);
//     simple_string& assign(const charT* s);
//     simple_string& assign(size_type n, charT c);
//     template<class InputIterator>
//     simple_string& assign(InputIterator first, InputIterator last);
    
    static char sequence[] = "blub"; // COMPILER BUG: should not be `const char'

    simple_string s1;                          // ""
    simple_string s2 = "abc";                  // "abc"
    simple_string s3(s2 + "def");              // "abcdef"
    simple_string s4(s3, 2, 2);                // "cd"
    simple_string s5(s3, 6, 99);               // ""
    simple_string s6(s3, 2);                   // "cdef"
    simple_string s7(s3, 2, 4);                // "cdef"
    simple_string s8("foobar", 4);             // "foob"
    simple_string s9(10, 'x');                 // "xxxxxxxxxx"
    simple_string s10(0, 'a');                 // ""
    simple_string s11(sequence, &sequence[4]); // "blub"

    TEST(std::strcmp(s1.c_str(), "") == 0);    
    TEST(std::strcmp(s2.c_str(), "abc") == 0);    
    TEST(std::strcmp(s3.c_str(), "abcdef") == 0);    
    TEST(std::strcmp(s4.c_str(), "cd") == 0);    
    TEST(std::strcmp(s5.c_str(), "") == 0);    
    TEST(std::strcmp(s6.c_str(), "cdef") == 0);    
    TEST(std::strcmp(s7.c_str(), "cdef") == 0);    
    TEST(std::strcmp(s8.c_str(), "foob") == 0);    
    TEST(std::strcmp(s9.c_str(), "xxxxxxxxxx") == 0);    
    TEST(std::strcmp(s10.c_str(), "") == 0);    
    TEST(std::strcmp(s11.c_str(), "blub") == 0);

    s1 = s6;
    TEST(std::strcmp(s1.c_str(), "cdef") == 0);
    TEST(std::strcmp(s6.c_str(), "cdef") == 0);
    s10 = "fasel";
    s5 = 'x';
    s11 = s2 + s3 + s4;
    TEST(std::strcmp(s10.c_str(), "fasel") == 0);    
    TEST(std::strcmp(s5.c_str(),  "x") == 0);    
    TEST(std::strcmp(s11.c_str(), "abcabcdefcd") == 0);    
    TEST(std::strcmp(s3.c_str(),  "abcdef") == 0);

    s1.assign(s2);                   // abc
    s2.assign(s2, 2, simple_string::npos); // c
    s3.assign(s11, 3, 3);            // abc
    s4.assign("quuxuum", 4);         // quux
    s5.assign("quuxuum");            // quuxuum
    s6.assign(7, '-');               // -------
    s7.assign(sequence, sequence+2); // bl

    TEST(std::strcmp(s1.c_str(), "abc") == 0);
    TEST(std::strcmp(s2.c_str(), "c") == 0);
    TEST(std::strcmp(s3.c_str(), "abc") == 0);
    TEST(std::strcmp(s4.c_str(), "quux") == 0);
    TEST(std::strcmp(s5.c_str(), "quuxuum") == 0);
    TEST(std::strcmp(s6.c_str(), "-------") == 0);
    TEST(std::strcmp(s7.c_str(), "bl") == 0);
}

/* test misc. accessor / manipulator functions */
static void
test_misc()
{
//     size_type length() const { return len; }
//     const charT* data() const { return txt; }
//     size_type size() const { return len; }
//     size_type capacity() const { return len; }
//     size_type max_size() const { return npos >> 1; /* be conservative */ }
//     const charT* c_str() const;
//     void resize(size_type n, charT c);
//     void resize(size_type n) { resize(n, charT()); }
//     void swap(simple_string& rhs);
//     size_type copy(charT* s, size_type n, size_type pos = 0) const;
//     reference operator[](size_type pos) { return txt[pos]; }
//     const_reference operator[](size_type pos) const { return pos == len ? null : txt[pos]; }
//     reference at(size_type pos);
//     const_reference at(size_type pos) const;
//     bool empty() const { return len==0; }
//     void clear();
//     int compare(const simple_string& str) const;
//     int compare(size_type pos1, size_type n1,
//                 const simple_string& str) const;
//     int compare(size_type pos1, size_type n1,
//                 const simple_string& str,
//                 size_type pos2, size_type n2) const;
//     int compare(const charT* s) const;
//     int compare(size_type pos1, size_type n1,
//                 const charT* s, size_type n2 = npos) const;
//     proxy substr(size_type pos = 0, size_type n = npos) const;
    simple_string s1 = "foobar";
    simple_string s2;
    TEST(s1.length() == 6);
    TEST(s2.length() == 0);
    TEST(s2.empty());
    TEST(std::memcmp(s1.data(), "foobar", 6) == 0);
    TEST(s1.size() == 6);
    TEST(s1.capacity() >= 6);
    TEST(std::strcmp(s1.c_str(), "foobar") == 0);
    TEST(s1 == "foobar");
    s1.resize(3);
    TEST(std::strcmp(s1.c_str(), "foo") == 0);
    TEST(s1 == "foo");
    s1.resize(5, 'x');
    TEST(std::strcmp(s1.c_str(), "fooxx") == 0);
    TEST(s1 == "fooxx");
    s1.swap(s2);
    TEST(s2 == "fooxx");
    TEST(s1.empty());
    {
        char buf[2];
        TEST(s2.copy(buf, 2, 3) == 2);
        TEST(buf[0] == 'x' && buf[1] == 'x');
        TEST(s1.copy(buf, 2, 0) == 0);
    }
    swap(s1, s2);
    TEST(s1 == "fooxx");
    TEST(s2.empty());
    TEST(const_cast<const simple_string&>(s2)[0] == 0);
    TEST(s1[0] == 'f' && s1[1] == 'o' && s1[3] == 'x');
    TEST(s1.at(0) == 'f' && s1.at(1) == 'o' && s1.at(3) == 'x');
    s1.at(4) = 'y';
    TEST(s1 == "fooxy");
    s1[1] = 'l';
    TEST(s1 == "floxy");
#ifndef BUILTIN_STRING
    /* Bug: std::string does not have clear */
    s1.clear();
#else
    s1.erase();
#endif
    TEST(s1.empty());
    TEST(s1 == s2);
    TEST(s1 == "");
    TEST("" == s1);

#ifndef BOGUS_STRING
    /* Bug: g++'s std::string's compare is broken beyond repair */
    TEST(s1.compare("foo") < 0);
    TEST(s1.compare("") == 0);
    TEST(s1.compare(s2) == 0);
    TEST(s1 < "foo");
    TEST(s1 == "");

    s1 = "pentagondodekaeder";
    TEST(s1.compare(3, 3, "tag", 3) == 0);
    // if I read the CD2 correctly, the following [should] be negative:
    // st.compare(3, 3, "tag") == string(*this, 3, 3).compare(string("tag", npos))
    // (i.e. it's actually undefined behaviour -- the default argument is
    // pretty misguiding)
    TEST(s1.compare(3, 3, "tag") < 0);
    TEST(s1.compare("xyzzy") < 0);
    TEST(s1.compare("dostoprimetschatjelnostj") > 0);
    TEST(s1.compare(1, 4, "enta", 4) == 0);
    TEST(s1.compare(1, 4, "ente", 4) < 0);
    s2 = "makkaroni";
    TEST(s1.compare(6, 2, s2, 6, 2) == 0);
    --s2[6];
    TEST(s1.compare(6, 2, s2, 6, 2) > 0);
    TEST(s1 < "xyzzy");
    TEST("xyzzy" > s1);
    TEST(s1 > "mslifresser");
    TEST(s1 != "mslifresser");
    TEST(s1.compare("pdagoge") < 0);
    TEST(s1.compare("pantoffel") > 0);
#endif

    s1 = "brennholzverleihsteuer";
    TEST(s1.substr(5, 4) == "holz");
    s2 = s1.substr(5, 4) + s1.substr(0, 5) + s1.substr(16);
    TEST(s2 == "holzbrennsteuer");
    s2 = simple_string(s1, 5, 4);
    TEST(s2 == "holz");

    TEST((s2 + "foo").length() == 7);
    TEST(std::strcmp((s2 + "foo").c_str(), "holzfoo") == 0);
    TEST(!(s2 + "foo").empty());
    TEST((simple_string() + "").empty());
    TEST((s2 + "foo")[5] == 'o');
#ifndef BOGUS_STRING
    TEST((s2 + "foo").compare(3, 2, "zf", 2) == 0);
#endif
}

/* test append functions */
static void
test_append()
{
//     simple_string& operator+=(const simple_string& rhs);
//     simple_string& operator+=(const charT* rhs);
//     simple_string& operator+=(charT rhs);
//     simple_string& append(const simple_string& str);
//     simple_string& append(const simple_string& str, size_type pos, size_type n);
//     simple_string& append(const charT* s, size_type n);
//     simple_string& append(const charT* s);
//     simple_string& append(size_type n, charT c);
//     template<class InputIterator>
//     simple_string& append(InputIterator first, InputIterator last);
    simple_string s1 = "foo", s2 = "bar", s3;
    TEST(s1 == "foo");
    TEST(s2 == "bar");
    TEST(s3.empty());

    s1 += s2;
    TEST(s1 == "foobar");
    TEST(s2 == "bar");
    s1 += "quux";
    TEST(s1 == "foobarquux");
    s1 += '!';
    TEST(s1 == "foobarquux!");
    s1 = "foo";
    TEST(s1 == "foo");
    s1.append(s2);
    TEST(s1 == "foobar");
    s1.append(s1, 3, simple_string::npos);
    TEST(s1 == "foobarbar");
    s1.append(s1, 0, 3);
    TEST(s1 == "foobarbarfoo");
    s2.append("hualp", 2);
    TEST(s2 == "barhu");
    s2.append("ps");
    TEST(s2 == "barhups");
    s2.append(3, '!');
    TEST(s2 == "barhups!!!");
    s3.append(4, 'x');
    TEST(s3 == "xxxx");
}

/* test erase / replace */
static void
test_erase_replace()
{
//     simple_string& erase(size_type pos = 0, size_type n = npos);
//     iterator erase(iterator position);
//     iterator erase(iterator first, iterator last);

//     simple_string& replace(size_type pos1, size_type n1,
//                            const simple_string& str);
//     simple_string& replace(size_type pos1, size_type n1,
//                            const simple_string& str,
//                            size_type pos2, size_type n2);
//     simple_string& replace(size_type pos, size_type n1, const charT* s, size_type n2);
//     simple_string& replace(size_type pos, size_type n1, const charT* s);
//     simple_string& replace(size_type pos, size_type n1, size_type n2, charT c);
//     simple_string& replace(iterator i1, iterator i2, const simple_string& str);
//     simple_string& replace(iterator i1, iterator i2, const charT* s, size_type n);
//     simple_string& replace(iterator i1, iterator i2, const charT* s);
//     simple_string& replace(iterator i1, iterator i2, size_type n, charT c);
//     template<class InputIterator>
//     simple_string& replace(iterator i1, iterator i2,
//                            InputIterator j1, InputIterator j2) {
//         return replace(i1, i2, simple_string(j1, j2));
//     }
//     simple_string& insert(size_type pos1, const simple_string& str);
//     simple_string& insert(size_type pos1, const simple_string& str,
//                           size_type pos2, size_type n);
//     simple_string& insert(size_type pos, const charT* s, size_type n);
//     simple_string& insert(size_type pos, const charT* s);
//     simple_string& insert(size_type pos, size_type n, charT c);
//     iterator insert(iterator p, charT c = charT());
//     void insert(iterator p, size_type n, charT c);
//     template<class InputIterator>
//     void insert(iterator p, InputIterator first, InputIterator last) {
//         insert(p - begin(), simple_string(first, last));
//     }
    simple_string s1, s2, s3;
    s1 = "foobar";
    TEST(s1 == "foobar");
    s1.erase(s1.begin());
    TEST(s1 == "oobar");
    s1.erase();
    TEST(s1.empty());
    TEST(s1 == "");
    TEST(s1 == simple_string());
    s1 = "foobar";
    s1.erase(s1.begin()+1, s1.begin()+3);
    TEST(s1 == "fbar");
    s1.erase(0, 1).erase(1, 1);
    TEST(s1 == "br");
    s2 = "arpawocky";
    TEST(s2.length() == 9);
    s2.erase(4);
    TEST(s2 == "arpa");
    s2.replace(1, 2, "bb");
    TEST(s2 == "abba");
    s2.replace(0, 2, "abracadafasel", 8);
    TEST(s2 == "abracadaba");
    s2.replace(8, 1, s1);
    TEST(s2 == "abracadabra");
    s1.replace(0, 1, s2, 4, 3);
    TEST(s1 == "cadr");
    s1.replace(2, 1, 10, 'x');
    TEST(s1 == "caxxxxxxxxxxr");
    s1.replace(s1.begin(), s1.end() - 4, "hooray");
    TEST(s1 == "hoorayxxxr");
    s1.replace(s1.begin(), s1.begin()+3, "x-y-ungelst", 2);
    TEST(s1 == "x-rayxxxr");
    s1.erase(5);
    TEST(s1 == "x-ray");
    s1.insert(2, "y-");
    TEST(s1 == "x-y-ray");
    s1.insert(1, 3, '+');
    TEST(s1 == "x+++-y-ray");
    s1 = "foo";
    s2 = "bar";
    TEST(s1 == "foo" && s2 == "bar");
    s1.insert(2, s2);
    TEST(s1 == "fobaro");
    s2.insert(2, s1, 0, 2);
    TEST(s2 == "bafor");
    s2.insert(3, "weoiztsdgklvh", 2);
    TEST(s2 == "bafweor");
}

static void
test_find()
{
//     size_type find (const simple_string& str, size_type pos = 0) const;
//     size_type find (const charT* s, size_type pos, size_type n) const;
//     size_type find (const charT* s, size_type pos = 0) const;
//     size_type find (charT c, size_type pos = 0) const;

//     size_type rfind(const simple_string& str, size_type pos = npos) const;
//     size_type rfind(const charT* s, size_type pos, size_type n) const;
//     size_type rfind(const charT* s, size_type pos = npos) const;
//     size_type rfind(charT c, size_type pos = npos) const;

//     size_type find_first_of(const simple_string& str, size_type pos = 0) const;
//     size_type find_first_of(const charT* s, size_type pos, size_type n) const;
//     size_type find_first_of(const charT* s, size_type pos = 0) const;
//     size_type find_first_of(charT c, size_type pos = 0) const;

//     size_type find_last_of(const simple_string& str,
//                            size_type pos = npos) const;
//     size_type find_last_of(const charT* s, size_type pos, size_type n) const;
//     size_type find_last_of(const charT* s, size_type pos = npos) const;
//     size_type find_last_of(charT c, size_type pos = npos) const;
    
//     size_type find_first_not_of(const simple_string& str, size_type pos = 0) const;
//     size_type find_first_not_of(const charT* s, size_type pos, size_type n) const;
//     size_type find_first_not_of(const charT* s, size_type pos = 0) const;
//     size_type find_first_not_of(charT c, size_type pos = 0) const;

//     size_type find_last_not_of(const simple_string& str,
//                                size_type pos = npos) const;
//     size_type find_last_not_of(const charT* s, size_type pos, size_type n) const;
//     size_type find_last_not_of(const charT* s, size_type pos = npos) const;
//     size_type find_last_not_of(charT c, size_type pos = npos) const;
    simple_string s1 = "the quick brown fox jumps over the lazy dog";
    simple_string s2 = "foobar";
    simple_string s3("dog", 4);

    TEST(s3.length() == 4);
    TEST(s1.find("quick") == 4);
    TEST(s1.find("quickie", 0, 5) == 4);
    TEST(s1.find("quickie") == simple_string::npos);
    TEST(s1.find('e') == 2);
    TEST(s1.find('e', 10) == 28);
    TEST(s1.find("the") == 0);
    TEST(s1.find("the", 10) == 31);
    TEST(s1.find("the", 31) == 31);
    TEST(s1.find("the", 32) == simple_string::npos);
    TEST(s1.find("the", 42) == simple_string::npos);
    TEST(s1.find(s3) == simple_string::npos);
    TEST(s1.find(s3.substr(0, 3)) == 40);
    TEST(s3.find(char(0)) == 3);

    TEST(s1.rfind("quick") == 4);
    TEST(s1.rfind("quickie", simple_string::npos, 5) == 4);
    TEST(s1.rfind("quickie") == simple_string::npos);
    TEST(s1.rfind('e') == 33);
    TEST(s1.rfind('e', 10) == 2);
    TEST(s1.rfind("the") == 31);
    TEST(s1.rfind("the", 10) == 0);
    TEST(s1.rfind("the", 31) == 31);
    TEST(s1.rfind(s3) == simple_string::npos);
    TEST(s1.rfind(s3.substr(0, 3)) == 40);
    TEST(s3.rfind(char(0)) == 3);

    // simple_string s1 = "the quick brown fox jumps over the lazy dog";
    // simple_string s2 = "foobar";
    TEST(s1.find_first_of(s2) == 10);      // 'b'
    TEST(s1.find_first_of(s2, 11) == 11);  // 'r'
    TEST(s1.find_first_of("aeiou") == 2);
    TEST(s1.find_first_of("aeiou", 10) == 12);
    TEST(s1.find_first_of("BIGfoobar", 0, 3) == simple_string::npos);
    TEST(s1.find_first_of("BIGfoobar", 0) == 10);
    TEST(s1.find_first_of('e') == 2);
    TEST(s1.find_first_of('e', 10) == 28);

    TEST(s1.find_last_of(s2) == 41);      // 'o'
    TEST(s1.find_last_of(s2, 11) == 11);  // 'r'
    TEST(s1.find_last_of("aeiou") == 41);
    TEST(s1.find_last_of("aeiou", 10) == 6);
    TEST(s1.find_last_of("BIGfoobar", simple_string::npos, 3) == simple_string::npos);
    TEST(s1.find_last_of("BIGfoobar", simple_string::npos) == 41);
    TEST(s1.find_last_of('e') == 33);
    TEST(s1.find_last_of('e', 10) == 2);

    // simple_string s1 = "the quick brown fox jumps over the lazy dog";
    // simple_string s2 = "foobar";
    TEST(s1.find_first_not_of("the quick") == 10);
    TEST(s1.find_first_not_of("the quick", 10) == 10);
    TEST(s2.find_first_not_of(s1) == simple_string::npos);
    TEST(s1.find_first_not_of('t') == 1);
    TEST(s1.find_first_not_of("the quick", 0, 4) == 4);
    TEST(s1.find_first_not_of('q', 5) == 5);
    
    TEST(s1.find_last_not_of("lazy dog") == 33);
    TEST(s1[33] == 'e');
    TEST(s1.find_last_not_of("lazy dog", 37) == 33);
    TEST(s2.find_last_not_of(s1) == simple_string::npos);
    TEST(s2.find_last_not_of(s1.substr(10, 2)) == 4);
    TEST(s1.find_last_not_of('o') == 42);
    TEST(s1.find_last_not_of('o', 42) == 42);
    TEST(s1.find_last_not_of('o', 41) == 40);
    TEST(s1.find_last_not_of("dog lazy", simple_string::npos, 4) == 38);

    s1 = "pentgnddekeder";
    TEST(s1.find('') == 4);
    TEST(s1.find("ke") == 12);
    TEST(s1.find_first_not_of("gpent") == 8);
    TEST(s1.find('', 9) == 13);
}

int main(int argc, char** argv)
{
    int count = argv[1] ? std::atoi(argv[1]) : 1;
    if (count <= 0) {
        std::cerr << "usage: tests [count]\n";
        return 1;
    }
#ifdef BUILTIN_STRING
    std::cout << "Testing std::string..." << std::endl;
#else
    std::cout << "Testing simple_string..." << std::endl;
#endif
    while (--count >= 0) {
        test_init();
        test_misc();
        test_append();
        test_erase_replace();
        test_find();
    }
    std::cout << "Done." << std::endl;
}
