/**
  *  \file testsig.cc
  */

#include "cpluslib/sigslot.h"

#include <iostream>
#include "cpluslib/assert.h"

/***************************** Parameterless *****************************/

class Receiver {
 public:
    int i;
    SignalConnection conn;
    Receiver(int i, Signal0& sig)
        : i(i), conn(sig.add(this, &Receiver::handle))
        { }
    void handle()
        { std::cout << "Receiver(" << i << ").handle()\n"; }
};

void
staticFun()
{
    std::cout << "staticFun()\n";
}

void
testFirst()
{
    std::cout << "-- explicit signal connect/disconnect; disconnect first handler --\n";
    Signal0 sig;
    {
        Receiver rec(0, sig);
        sig.add(staticFun);
        std::cout << "> should see two handlers:\n";
        sig.raise();
    }
    std::cout << "> should see one handler:\n";
    sig.raise();
}

void
testSecond()
{
    std::cout << "-- explicit signal connect/disconnect; disconnect second handler --\n";
    Signal0 sig;
    {
        sig.add(staticFun);
        Receiver rec(0, sig);
        std::cout << "> should see two handlers:\n";
        sig.raise();
    }
    std::cout << "> should see one handler:\n";
    sig.raise();
}

void
testThird()
{
    std::cout << "-- signal dies before receiver --\n";
    Signal0* sig = new Signal0;
    Receiver rec(0, *sig);
    std::cout << "rec.conn.isConnected() = " << rec.conn.isConnected() << "\n";
    sig->raise();
    delete sig;
    std::cout << "rec.conn.isConnected() = " << rec.conn.isConnected() << "\n";
}

/***************************** With Argument *****************************/

class ReceiverArg {
 public:
    int i;
    SignalConnection conn;
    ReceiverArg(int i, Signal1<int>& sig)
        : i(i), conn(sig.add(this, &ReceiverArg::handle))
        { }
    void handle(int n)
        { std::cout << "Receiver(" << i << ").handle(" << n << ")\n"; }
};

void
staticFunArg(int n)
{
    std::cout << "staticFunArg(" << n << ")\n";
}

void
testFirstArg()
{
    std::cout << "-- explicit signal connect/disconnect; disconnect first handler; with arg --\n";
    Signal1<int> sig;
    {
        ReceiverArg rec(0, sig);
        sig.add(staticFunArg);
        std::cout << "> should see two handlers:\n";
        sig.raise(99);
    }
    std::cout << "> should see one handler:\n";
    sig.raise(88);
}

void
testSecondArg()
{
    std::cout << "-- explicit signal connect/disconnect; disconnect second handler; with arg --\n";
    Signal1<int> sig;
    {
        // we use staticFun here to test invocation without args
        sig.add(staticFun);
        ReceiverArg rec(0, sig);
        std::cout << "> should see two handlers:\n";
        sig.raise(99);
    }
    std::cout << "> should see one handler:\n";
    sig.raise(88);
}

void
testThirdArg()
{
    std::cout << "-- signal dies before receiver --\n";
    Signal1<int>* sig = new Signal1<int>;
    ReceiverArg rec(0, *sig);
    std::cout << "rec.conn.isConnected() = " << rec.conn.isConnected() << "\n";
    sig->raise(77);
    delete sig;
    std::cout << "rec.conn.isConnected() = " << rec.conn.isConnected() << "\n";
}

void
testMulti()
{
    std::cout << "-- four identical handlers --\n";
    Signal0 sig;
    sig.add(staticFun);
    sig.add(staticFun);
    sig.add(staticFun);
    sig.add(staticFun);
    sig.raise();
}

void
testMultiRec()
{
    std::cout << "-- four objects --\n";
    Signal0 sig;
    Receiver r1(0, sig), r2(1, sig), r3(2, sig), r4(3, sig);
    sig.raise();
}

int
main()
{
    testFirst();
    testSecond();
    testThird();
    testFirstArg();
    testSecondArg();
    testThirdArg();
    testMulti();
    testMultiRec();
}
