/**
  *  \file substr.cc
  *  \brief Operations on Substrings
  *  \author Stefan Reuther
  *
  *  (c) 2001-2005 Stefan Reuther <Streu@gmx.de>
  *
  *  This program is free software; you can redistribute it and/or
  *  modify it under the terms of file `COPYING' that comes with the
  *  source code.
  *
  *  This program is distributed in the hope that it will be useful,
  *  but WITHOUT ANY WARRANTY; without even the implied warranty of
  *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  */

#include <cstring>
#include <cctype>
#include "cpluslib/substr.h"
#include "cpluslib/assert.h"
#include "cpluslib/cstring.h"

using namespace std;

#ifdef STREU_CONFIG_USE_STD_STRING
#  define data_or_null data
#endif

template<class T>
static inline T
xstr_min(T a, T b)
{
    return a < b ? a : b;
}

const substring::size_type substring::npos;

/** Construct from (static) C string. Note that, until the substring
    is assigned a different value, the C string must remain
    allocated. */
substring::substring(const charT* src)
    : str(src), len(strlen(src))
{ }

/** Construct from C++ string. Note that, until the substring is
    assigned a different value, the C++ string must remain unchanged
    and not die. */
substring::substring(const string_t& str, size_type pos, size_type n)
{
    ASSERT(pos <= str.size());
    this->str = str.data_or_null() + pos;
    this->len = xstr_min(str.size() - pos, n);
}

/** Construct from a substring. */
substring::substring(const substring& str, size_type pos, size_type n)
{
    ASSERT(pos <= str.size());
    this->str = str.data_or_null() + pos;
    this->len = xstr_min(str.size() - pos, n);
}

/** Construct from a char array. */
substring::substring(const charT src[], size_type n)
    : str(src), len(n)
{ }

/** Swap two substrings. */
void
substring::swap(substring& rhs)
{
    substring tmp = *this;
    *this = rhs;
    rhs = tmp;
}

/** Copy part of this string into a char array.
    \param s      [out] Array
    \param n      Maximum number of bytes to copy
    \param pos    Starting position
    \return number of bytes copied */
substring::size_type
substring::copy(charT s[], size_type n, size_type pos) const
{
    ASSERT(pos <= length());
    size_type rlen = xstr_min(n, length() - pos);
    memcpy(s, str + pos, rlen);
    return rlen;
}

/** Get character by index. */
substring::const_reference
substring::at(size_type pos) const
{
    ASSERT(pos < length());
    return str[pos];
}

/** Assign substring. */
substring&
substring::assign(const substring& str, size_type pos, size_type n)
{
    assign(substring(str, pos, n));
    return *this;
}

/** Assign C++ string. */
substring&
substring::assign(const string_t& str)
{
    assign(substring(str));
    return *this;
}

/** Assign substring of a C++ string. */
substring&
substring::assign(const string_t& str, size_type pos, size_type n)
{
    assign(substring(str, pos, n));
    return *this;
}

/** Assign C string. */
substring&
substring::assign(const charT* s)
{
    assign(substring(s));
    return *this;
}

/** Compare to C++ string. */
int
substring::compare(const substring& str) const
{
    return cstring_util::compare(data_or_null(), length(),
                                 str.data_or_null(), length());
}

/** Compare to substring. */
int
substring::compare(size_type pos1, size_type n1, const substring& str) const
{
    return cstring_util::compare(data_or_null() + pos1,
                                 xstr_min(n1, length() - pos1),
                                 str.data_or_null(),
                                 length());
}

int
substring::compare(size_type pos1, size_type n1,
                   const substring& str,
                   size_type pos2, size_type n2) const
{
    return cstring_util::compare(data_or_null() + pos1,
                                 xstr_min(n1, length() - pos1),
                                 str.data_or_null() + pos2,
                                 xstr_min(n2, str.length() - pos2));
}

int
substring::compare(const charT* s) const
{
    return cstring_util::compare(data_or_null(), length(),
                                 s, std::strlen(s));
}

int
substring::compare(size_type pos1, size_type n1,
                   const charT* s, size_type n2) const
{
    return cstring_util::compare(data_or_null() + pos1,
                                 xstr_min(n1, length() - pos1),
                                 s, n2);
}

substring
substring::substr(size_type pos, size_type n) const
{
    return substring(data_or_null() + pos, xstr_min(n, length() - pos));
}

substring::size_type
substring::find (const substring& str, size_type pos) const
{
    return cstring_util::find(data_or_null(), length(), pos,
                              str.data_or_null(), str.length());
}

substring::size_type
substring::find (const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find(data_or_null(), length(), pos, s, n);
}

substring::size_type
substring::find (const charT* s, size_type pos) const
{
    return cstring_util::find(data_or_null(), length(), pos,
                              s, std::strlen(s));
}

substring::size_type
substring::find (charT c, size_type pos) const
{
    return cstring_util::find(data_or_null(), length(), pos, c);
}

substring::size_type
substring::rfind(const substring& str, size_type pos) const
{
    return cstring_util::rfind(data_or_null(), length(), pos,
                               str.data_or_null(), str.length());
}

substring::size_type
substring::rfind(const charT* s, size_type pos, size_type n) const
{
    return cstring_util::rfind(data_or_null(), length(), pos, s, n);
}

substring::size_type
substring::rfind(const charT* s, size_type pos) const
{
    return cstring_util::rfind(data_or_null(), length(), pos,
                               s, std::strlen(s));
}

substring::size_type
substring::rfind(charT c, size_type pos) const
{
    return cstring_util::rfind(data_or_null(), length(), pos, c);
}

substring::size_type
substring::find_first_of(const substring& str, size_type pos) const
{
    return cstring_util::find_first_of(data_or_null(), length(), pos,
                                       str.data_or_null(), str.length());
}

substring::size_type
substring::find_first_of(const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find_first_of(data_or_null(), length(), pos, s, n);
}

substring::size_type
substring::find_first_of(const charT* s, size_type pos) const
{
    return cstring_util::find_first_of(data_or_null(), length(), pos,
                                       s, std::strlen(s));
}

substring::size_type
substring::find_first_of(charT c, size_type pos) const
{
    return cstring_util::find(data_or_null(), length(), pos, c);
}

substring::size_type
substring::find_last_of(const substring& str, size_type pos) const
{
    return cstring_util::find_last_of(data_or_null(), length(), pos,
                                      str.data_or_null(), str.length());
}

substring::size_type
substring::find_last_of(const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find_last_of(data_or_null(), length(), pos,
                                      s, n);
}

substring::size_type
substring::find_last_of(const charT* s, size_type pos) const
{
    return cstring_util::find_last_of(data_or_null(), length(), pos,
                                      s, std::strlen(s));
}

substring::size_type
substring::find_last_of(charT c, size_type pos) const
{
    return cstring_util::rfind(data_or_null(), length(), pos, c);
}

substring::size_type
substring::find_first_not_of(const substring& str, size_type pos) const
{
    return cstring_util::find_first_not_of(data_or_null(), length(), pos,
                                           str.data_or_null(), str.length());
}

substring::size_type
substring::find_first_not_of(const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find_first_not_of(data_or_null(), length(), pos,
                                           s, n);
}

substring::size_type
substring::find_first_not_of(const charT* s, size_type pos) const
{
    return cstring_util::find_first_not_of(data_or_null(), length(), pos,
                                           s, std::strlen(s));
}

substring::size_type
substring::find_first_not_of(charT c, size_type pos) const
{
    return cstring_util::find_first_not_of(data_or_null(), length(), pos,
                                           &c, 1);
}

substring::size_type
substring::find_last_not_of(const substring& str, size_type pos) const
{
    return cstring_util::find_last_not_of(data_or_null(), length(), pos,
                                          str.data_or_null(), length());
}

substring::size_type
substring::find_last_not_of(const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find_last_not_of(data_or_null(), length(), pos, s, n);
}

substring::size_type
substring::find_last_not_of(const charT* s, size_type pos) const
{
    return cstring_util::find_last_not_of(data_or_null(), length(), pos,
                                          s, std::strlen(s));
}

substring::size_type
substring::find_last_not_of(charT c, size_type pos) const
{
    return cstring_util::find_last_not_of(data_or_null(), length(), pos,
                                          &c, 1);
}

/** Trim leading and trailing whitespace. */
substring
strTrim(substring s)
{
    return strRTrim(strLTrim(s));
}

/** Trim leading (left) whitespace. */
substring
strLTrim(substring s)
{
    substring::size_type n = 0;
    while (n < s.length() && isspace(s[n]))
        ++n;
    return s.substr(n);
}

/** Trim trailing (right) whitespace. */
substring
strRTrim(substring s)
{
    substring::size_type n = s.length();
    while (n > 0 && isspace(s[n-1]))
        --n;
    return s.substr(0, n);
}

/** Return leading text up to separator. If there is no separator,
    return whole string. This can be used for splitting strings into
    pieces:
    \code
      string_t str = ...
      do {
         process(strFirst(str, ":"));
      } while(strRemove(std, ":"));
    \endcode */
substring
strFirst(substring str, substring sep)
{
    substring::size_type n = str.find(sep);
    if (n == substring::npos)
        return str;
    else
        return str.substr(0, n);
}

/** Remove leading text up to separator.
    Returns true iff ok, false if the string does not contain a separator.
    \see strFirst */
bool
strRemove(substring& str, const substring sep)
{
    substring::size_type n = str.find(sep);
    if (n == substring::npos) {
        return false;
    } else {
        str.assign(str, n + sep.length(), substring::npos);
        return true;
    }
}

/** Split string at separator.
    \param str      [in] string
    \param lhs,rhs  [out] place for output
    \param sep      [in] separator
    \retval true if separator found, in this case lhs+sep+rhs == str
    \retval false if separator not found, in this case lhs/rhs unmodified */
bool
strSplit(const substring str, substring& lhs, substring& rhs, const substring sep)
{
    substring::size_type n = str.find(sep);
    if (n == substring::npos) {
        return false;
    } else {
        lhs = str.substr(0, n);
        rhs = str.substr(n+sep.length());
        return true;
    }
}
