/**
  *  \file string.cc
  *  \brief String Utilities
  *  \author Stefan Reuther
  *
  *  This file contains a number of useful string operations. All
  *  functions operate on string_t, the preferred string type as
  *  selected in cpluslib/config.h.
  *
  *  (c) 2001-2005 Stefan Reuther <Streu@gmx.de>
  *
  *  This program is free software; you can redistribute it and/or
  *  modify it under the terms of file `COPYING' that comes with the
  *  source code.
  *
  *  This program is distributed in the hope that it will be useful,
  *  but WITHOUT ANY WARRANTY; without even the implied warranty of
  *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  */
#include <cstdio>
#include <cctype>
#include <csignal>
#include <cstddef>
#include "cpluslib/string.h"

using namespace std;            // some compilers choke on std::toupper etc.

/** Compare two character arrays, ignoring case.
    \param a,na [in]  First array
    \param b,nb [in]  Second array

    \return 0 if arrays are equal, negative if a precedes b, positive
    if a follows b lexicographically. */
int
strCaseCmp(const char* a, std::size_t na, const char* b, std::size_t nb)
{
    while (na && nb) {
        if (int diff = toupper((unsigned char)(*a)) - toupper((unsigned char)(*b)))
            return diff;
        ++a, ++b, --na, --nb;
    }
    if (na)
        return 1;
    if (nb)
        return -1;
    return 0;
}

/** Compare two C strings, ignoring case.
    \param a [in] First C string
    \param b [in] Second C string
    \return 0 if equal, negative if a precedes b, positive if a follows b */
int
strCaseCmp(const char* a, const char* b)
{
    register unsigned char ac, bc;
    while ((ac = toupper((unsigned char)(*a))) == (bc = toupper((unsigned char)(*b))) && ac)
        ++a, ++b;
    return ac - bc;
}

/** Compare two C++ strings, ignoring case.
    \param a [in] First string
    \param b [in] Second string
    \return 0 if equal, negative if a precedes b, positive if a follows b */
int
strCaseCmp(const string_t& a, const string_t& b)
{
    return strCaseCmp(a.data(), a.size(), b.data(), b.size());
}

/** Compare two strings, ignoring case.
    \param a [in] First string, C-style
    \param b [in] Second string
    \return 0 if equal, negative if a precedes b, positive if a follows b */
int
strCaseCmp(const char* a, const string_t& b)
{
    return strCaseCmp(a, strlen(a), b.data(), b.size());
}

/** Compare two strings, ignoring case.
    \param a [in] First string
    \param b [in] Second string, C-style
    \return 0 if equal, negative if a precedes b, positive if a follows b */
int
strCaseCmp(const string_t& a, const char* b)
{
    return strCaseCmp(a.data(), a.size(), b, strlen(b));
}

/** Trim whitespace. Removes whitespace from both sides.
    \param s [in] Input
    \return s, with leading and trailing whitespace removed. */
string_t
strTrim(string_t s)
{
    return strLTrim(strRTrim(s));
}

/** Trim leading (left) whitespace.
    \param s [in] Input
    \return s, with leading whitespace (see #isspace) removed. */
string_t
strLTrim(string_t s)
{
    string_t::size_type n = 0;
    while (n < s.length() && isspace(s[n]))
        ++n;
    return s.substr(n);
}

/** Trim trailing (right) whitespace.
    \param s [in] Input
    \return s, with trailing whitespace (see #isspace) removed. */
string_t
strRTrim(string_t s)
{
    string_t::size_type n = s.length();
    while (n > 0 && isspace(s[n-1]))
        --n;
    return s.substr(0, n);
}

/** Convert string to lower case.
    \return new string. */
string_t
strLCase(string_t s)
{
    for (string_t::size_type n = 0; n < s.length(); ++n)
        s[n] = tolower((unsigned char) s[n]);
    return s;
}

/** Convert string to upper case.
    \return new string. */
string_t
strUCase(string_t s)
{
    for (string_t::size_type n = 0; n < s.length(); ++n)
        s[n] = toupper((unsigned char) s[n]);
    return s;
}

/** Capitalize string. Converts all letters following a space or
    hyphen to upper case, and others to lower case. */
string_t
strUCFirst(string_t s)
{
    bool uc = true;
    for (string_t::size_type n = 0; n < s.length(); ++n) {
        if (isspace(s[n]) || s[n] == '-') {
            uc = true;
        } else {
            if (uc)
                s[n] = toupper((unsigned char) s[n]);
            else
                s[n] = tolower((unsigned char) s[n]);
            uc = false;
        }
    }
    return s;
}

/** Un-Capitalize string. Converts a string in all-caps such that only
    word beginnings remain in upper-case. */
string_t
strLCWords(string_t s)
{
    bool uc = true;
    for (string_t::size_type n = 0; n < s.length(); ++n) {
        if (isspace(s[n]) || s[n] == '-') {
            uc = true;
        } else {
            if (!uc)
                s[n] = tolower((unsigned char) s[n]);
            uc = false;
        }
    }
    return s;
}

/** Convert integer to string. Returns the number string. */
string_t
itoa(long l)
{
    char buffer[100];
    std::sprintf(buffer, "%ld", l);
    return string_t(buffer);
}

/** Convert integer to string, adding a unit when appropriate.
    \overload */
string_t
itoa(long l, const char* a, const char* b)
{
    if (l % 101 == 1)
        return itoa(l) + a;
    else
        return itoa(l) + b;
}

/** Convert integer to string, adding a unit when appropriate.
    \param l   value
    \param a   singular unit ("ton")
    \param b   plural unit ("tons") */
string_t
itoa(long l, const string_t a, const string_t b)
{
    if (l % 101 == 1)
        return itoa(l) + a;
    else
        return itoa(l) + b;
}

/** Indent string with prefix. Prepend \c prefix to all lines in \c
    str. Lines are determined by '\n'. The first line itself is not
    indented. */
string_t
strIndent(const string_t& prefix, string_t str)
{
    string_t::size_type p = string_t::npos;
    while (1) {
        p = str.rfind('\n', p);
        if (p == string_t::npos)
            return str;
        str.insert(p+1, prefix);
        if (!p)
            return str;
        --p;
    }
}

/** Return leading text up to separator. If there is no separator,
    returns the whole string. This can be used for splitting strings
    into pieces:
    \code
      string_t str = ...
      do {
         process(strFirst(str, ":"));
      } while(strRemove(std, ":"));
    \endcode */
string_t
strFirst(const string_t str, const string_t sep)
{
    string_t::size_type n = str.find(sep);
    if (n == string_t::npos)
        return str;
    else
        return str.substr(0, n);
}

/** Remove leading text up to separator.
    Returns true iff ok, false if the string does not contain a separator.
    \param str        [in/out] String. It will be modified, everything
                      up to the first occurence of \c sep will be removed
    \param sep        [in] separator
    \see strFirst */
bool
strRemove(string_t& str, const string_t sep)
{
    string_t::size_type n = str.find(sep);
    if (n == string_t::npos) {
        return false;
    } else {
        str.erase(0, n + sep.length());
        return true;
    }
}

/** Split string at separator.
    \param str      [in] string
    \param lhs,rhs  [out] place for output
    \param sep      [in] separator
    \retval true if separator found, in this case lhs+sep+rhs == str
    \retval false if separator not found, in this case lhs/rhs unmodified */
bool
strSplit(const string_t str, string_t& lhs, string_t& rhs, const string_t sep)
{
    string_t::size_type n = str.find(sep);
    if (n == string_t::npos) {
        return false;
    } else {
        lhs = str.substr(0, n);
        rhs = str.substr(n+sep.length());
        return true;
    }
}
