/**
  *  \file sstring.cc
  *  \brief Small String
  *
  *  A string implementation intended to generate small code.
  *
  *  (c) 2001-2005 Stefan Reuther <Streu@gmx.de>
  *
  *  This program is free software; you can redistribute it and/or
  *  modify it under the terms of file `COPYING' that comes with the
  *  source code.
  *
  *  This program is distributed in the hope that it will be useful,
  *  but WITHOUT ANY WARRANTY; without even the implied warranty of
  *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  */

#include <cstring>
#include "cpluslib/sstring.h"
#include "cpluslib/assert.h"
#include "cpluslib/cstring.h"
#include "cpluslib/substr.h"

/** Allocation Policy. */
enum {
    /** Above ALLOC_THRESHOLD, we add ALLOC_INC. Below, we double. */
    SSTRING_ALLOC_THRESHOLD = 8191,
    SSTRING_ALLOC_INC       = 128,
    /** We allocate no less than this. */
    SSTRING_ALLOC_MIN       = 15
};

/*! \class simple_string
    \brief Simple String Class

    A string class which, unlike std::string, generates very tight
    code for most operations by avoiding temporaries with destructors.
    These generate a lot of overhead for the unlikely case of an
    exception (test suite with simple_string: 30k, with std::string: 45k).

    This class is otherwise (intended to be) exactly like std::string,
    with the following exceptions:
    - it's not a template. Rationale: allows putting implementation code
      in a .cc file.
    - allocators are not implemented. Rationale: no templates.
    - certain operations which require simple_string objects do not
      work with simple_string expressions because of the proxy objects
      (e.g. a `class foo' with a ctor `foo::foo(simple_string)',
      and a function `foothing(foo)', the function can not be
      called as `foothing(a + b)'; with std::string, this would work.
      Workaround: add a cast: `foothing(simple_string(a + b))'
    - a simple_string must not be modified while an expression referring
      to it is being evaluated. I.e. `(x+="a") + (x+="b") + (x+="c")'
      does not work. This problem can be solved (by having the proxy
      objects point to the actual string objects and not their contents)
      but I think it's not worth it: `(i+=1) + (i+=2) + (i+=3)' with
      i being int has undefined behaviour, and nobody complains there.
    - this implementation does not fulfill the STL namespace rules because
      it uses names the user is allowed to \#define ;-)

    You should never create proxy objects manually.

    Implementation note: we always allocate one byte more than needed,
    to support an efficient/simple version of c_str(). When the string
    is empty, txt may be null.

    \todo
    - operator>>
    - optimisations:
      - compare(proxy)
      - append(proxy), operator+(string, proxy)
    - reverse iterator
    - do we need more test cases?
    - simple_stringbuf / simple_stringstream
    - length overflow not yet handled (i.e. taking a 1G string and doing
      `s + s + s')
    - inline some more functions?

    Implementor Guidelines:
    - Use C functions, not \<algorithm>. Rationale: no exceptions, speed
      (gcc automatically knows that C functions never throw) */

const char simple_string::null = 0;

template<class T>
static inline T
xstr_min(T a, T b)
{
    return a < b ? a : b;
}

/*********************** simple_string constructors **********************/

/** Copy constructor. Create a copy of \c rhs. */
simple_string::simple_string(const simple_string& rhs)
{
    if (rhs.len) {
        cap = len = rhs.len;
        txt = new charT[rhs.len+1];
        std::memcpy(txt, rhs.txt, len);
    } else {
        cap = len = 0;
        txt = 0;
    }
}

/** Initialize with C string. */
simple_string::simple_string(const charT* src)
{
    if (*src) {
        cap = len = std::strlen(src);
        txt = new charT[len+1];
        std::memcpy(txt, src, len);
    } else {
        cap = len = 0;
        txt = 0;
    }
}

/** Initialize with substring. Roughly equivalent to initializing with
    str.substr(pos, n) */
simple_string::simple_string(const simple_string& str, size_type pos,
                             size_type n)
{
    ASSERT(pos <= str.length());

    cap = len = xstr_min(str.length() - pos, n);
    if (len) {
        txt = new charT[len+1];
        std::memcpy(txt, str.txt + pos, len);
    } else {
        txt = 0;
    }
}

/** Initialize from substring object. */
simple_string::simple_string(const substring& str)
{
    if (str.length()) {
        cap = len = str.length();
        txt = new charT[len+1];
        std::memcpy(txt, str.data_or_null(), len);
    } else {
        cap = len = 0;
        txt = 0;
    }
}

/** Initialize from character array.
    \param src    array of characters
    \param n      number of characters to copy */
simple_string::simple_string(const charT* src, size_type n)
    : txt(new charT[n+1]), len(n), cap(n)
{
    std::memcpy(txt, src, n);
}

/** Init with n copies of character c. */
simple_string::simple_string(size_type n, charT c)
    : txt(new charT[n+1]), len(n), cap(n)
{
    std::memset(txt, c, n);
}

/*********************** init-from-proxy functions ***********************/

/** Initialize from a proxy object. */
void
simple_string::init_from_proxy(const proxy& rhs)
{
    sizeT newlen = rhs.length();

    if (newlen) {
        txt = new charT[newlen+1];
        cap = len = newlen;
        for (const proxy* p = &rhs; p; p = p->next)
            if (p->len)
                std::memcpy(&txt[newlen -= p->len], p->txt, p->len);
    } else {
        txt = 0;
        cap = len = 0;
    }
}

/** Assign from a proxy object. We have to be careful because the
    proxy might point into this string. */
void
simple_string::assign_from_proxy(const proxy& rhs)
{
    sizeT newlen = rhs.length();

    if (newlen) {
        charT* ntxt = new charT[newlen+1];
        sizeT nlen = newlen;
        for (const proxy* p = &rhs; p; p = p->next)
            if (p->len)
                std::memcpy(&ntxt[newlen -= p->len], p->txt, p->len);
        delete[] txt;
        txt = ntxt;
        cap = len = nlen;
    } else {
        delete[] txt;
        txt = 0;
        cap = len = 0;
    }
}

/***************************** erase methods *****************************/

/** Erase the n characters starting at pos.
    \return *this */
simple_string&
simple_string::erase(size_type pos, size_type n)
{
    replace(pos, n, (charT*) 0, 0);
    return *this;
}

/** Erase the character pointed-at by the iterator.
    \return iterator pointing after the just-deleted character */
simple_string::iterator
simple_string::erase(iterator position)
{
    size_type pos = position - begin();
    replace(pos, 1, (charT*) 0, 0);
    return begin() + pos;
}

/** Erase the characters in the given sequence.
    \return iterator pointing after the just-deleted sequence. */
simple_string::iterator
simple_string::erase(iterator first, iterator last)
{
    size_type pos = first - begin();
    replace(pos, last - first, (charT*) 0, 0);
    return begin() + pos;
}

/**************************** replace methods ****************************/

/** Replace subsequence with a string.
    \param pos1 first character to remove
    \param n1   number of characters to remove
    \param str  string to insert instead
    \return *this */
simple_string&
simple_string::replace(size_type pos1, size_type n1,
                       const simple_string& str)
{
    return replace(pos1, n1, str.data_or_null(), str.length());
}

/** Replace subsequence with substring. Same as
    replace(pos1, n1, str.substr(pos2, n2)).
    \return *this */
simple_string&
simple_string::replace(size_type pos1, size_type n1,
                       const simple_string& str,
                       size_type pos2, size_type n2)
{
    n2 = xstr_min(n2, str.length() - pos2);
    return replace(pos1, n1, str.data() + pos2, n2);
}

/** Replace subsequence with character array.
    Same as replace(pos, n1, simple_string(s, n2)).
    \return *this */
simple_string&
simple_string::replace(size_type pos, size_type n1,
                       const charT* s, size_type n2)
{
    ASSERT(pos <= length());

    if (n1 > length() - pos)
        n1 = length() - pos;

    if (len + n2 - n1 <= cap && (s < txt || s > txt + len)) {
        if (length() - pos - n1 && n1 != n2)
            std::memmove(txt + pos + n2, txt + pos + n1, length() - pos - n1);
        std::memcpy(txt + pos, s, n2);
        len = len + n2 - n1;
    } else {
        proxy2 px(0, data_or_null(), pos, s, n2);
        assign_from_proxy(proxy(&px, data_or_null() + pos + n1, length() - pos - n1));
    }
    return *this;
}

/** Replace subsequence with C-string.
    Same as replace(pos, n1, simple_string(s)).
    \return *this */
simple_string&
simple_string::replace(size_type pos, size_type n1, const charT* s)
{
    return replace(pos, n1, s, std::strlen(s));
}

/** Replace subsequence with n2 copies of c.
    Same as replace(pos, n1, simple_string(n2, c)).
    \return *this */
simple_string&
simple_string::replace(size_type pos, size_type n1, size_type n2, charT c)
{
    return replace(pos, n1, simple_string(n2, c));
}

/** Replace subsequence with string. Replaces everything in the given
    sequence with the string str.
    \return *this */
simple_string&
simple_string::replace(iterator i1, iterator i2, const simple_string& str)
{
    return replace(i1 - begin(), i2 - i1, str);
}

/** Replace subsequence with character array. Replaces everything in
    the given sequence with the the n bytes at s.
    \return *this */
simple_string&
simple_string::replace(iterator i1, iterator i2, const charT* s, size_type n)
{
    return replace(i1 - begin(), i2 - i1, s, n);
}

/** Replace subsequence with C-string. Replaces everything in the given
    sequence with the C-string s.
    \return *this */
simple_string&
simple_string::replace(iterator i1, iterator i2, const charT* s)
{
    return replace(i1 - begin(), i2 - i1, s);
}

/** Replace subsequence repeated character. Replaces everything in the
    given sequence with n copies of c.
    \return *this */
simple_string&
simple_string::replace(iterator i1, iterator i2, size_type n, charT c)
{
    return replace(i1 - begin(), i2 - i1, n, c);
}

/***************************** insert methods ****************************/

/** Insert string.
    \param pos1 where to insert
    \param str  what to insert
    \post  substr(pos1, str.length()) == str
    \return *this */
simple_string&
simple_string::insert(size_type pos1, const simple_string& str)
{
    return replace(pos1, 0, str);
}

/** Insert substring. Same as insert(pos1, str.substr(pos2, n)).
    \return *this */
simple_string&
simple_string::insert(size_type pos1, const simple_string& str,
                      size_type pos2, size_type n)
{
    return replace(pos1, 0, str, pos2, n);
}

/** Insert character array. Same as insert(pos1, simple_string(s, n)).
    \return *this */
simple_string&
simple_string::insert(size_type pos, const charT* s, size_type n)
{
    return replace(pos, 0, s, n);
}

/** Insert C-string. Same as insert(pos, simple_string(s)).
    \return *this */
simple_string&
simple_string::insert(size_type pos, const charT* s)
{
    return replace(pos, 0, s);
}

/** Insert repeated character. Inserts n copies of character c after
    position pos in this string.
    \return *this */
simple_string&
simple_string::insert(size_type pos, size_type n, charT c)
{
    return replace(pos, 0, n, c);
}

/** Insert character at position. Inserts c before iterator p.
    \return iterator pointing at just-inserted character. */
simple_string::iterator
simple_string::insert(iterator p, charT c)
{
    size_type pos = p - begin();
    insert(pos, 1, c);
    return begin() + pos;
}

/** Insert n copies of c at position. */
void
simple_string::insert(iterator p, size_type n, charT c)
{
    insert(p - begin(), n, c);
}

/**************************** compare methods ****************************/

/** Compare with string.
    \return 0 if equal, negative if *this<str, positive if *this>str */
int
simple_string::compare(const simple_string& str) const
{
    return cstring_util::compare(data_or_null(), length(),
                                 str.data_or_null(), str.length());
}

/** Compare substring with string. Same as substr(pos1, n1).compare(str).
    \return 0 if equal, negative if substring < str, positive if substring > str */
int
simple_string::compare(size_type pos1, size_type n1,
                       const simple_string& str) const
{
    return cstring_util::compare(data_or_null() + pos1,
                                 xstr_min(length()-pos1, n1),
                                 str.data_or_null(),
                                 str.length());
}

/** Compare substrings. Same as substr(pos1, n1).compare(str.substr(pos2, n2)). */
int
simple_string::compare(size_type pos1, size_type n1,
                       const simple_string& str,
                       size_type pos2, size_type n2) const
{
    return cstring_util::compare(data_or_null() + pos1,
                                 xstr_min(length()-pos1, n1),
                                 str.data_or_null() + pos2,
                                 xstr_min(str.length()-pos2, n2));
}

/** Compare with C string.
    \return 0 if equal, negative if *this<s, positive if *this>s */
int
simple_string::compare(const charT* s) const
{
    return cstring_util::compare(data_or_null(), length(),
                                 s, std::strlen(s));
}

/** Compare with char array.
    Same as substr(pos1, n1).compare(simple_string(s, n2)).
    Note that omitting n2 does not have the intended effect of treating
    s as a null-terminated string.
    \return 0 if equal, <0 if less, >0 if larger. */
int
simple_string::compare(size_type pos1, size_type n1,
                       const charT* s, size_type n2) const
{
    return cstring_util::compare(data_or_null() + pos1, xstr_min(length()-pos1, n1), s, n2);
}

/****************************** find methods *****************************/

/** Find substring. Finds the smallest position p such that p >= pos &&
    substr(pos, str.length()) == str.
    \return p, or npos if no such position exists. */
simple_string::size_type
simple_string::find (const simple_string& str, size_type pos) const
{
    return cstring_util::find(data_or_null(), length(), pos,
                              str.data_or_null(), str.length());
}

/** Find substring character array. Finds the smallest position p such that
    p >= pos and substr(pos, n) == simple_string(s, n).
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find (const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find(data_or_null(), length(), pos, s, n);
}

/** Find C-string. Finds the smallest position p such that
    p >= pos and substr(pos, strlen(s)) == simple_string(s).
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find (const charT* s, size_type pos) const
{
    return cstring_util::find(data_or_null(), length(), pos, s, std::strlen(s));
}

/** Find character. Finds the smallest position p such that
    p >= pos and at(p) == c.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find (charT c, size_type pos) const
{
    return cstring_util::find(data_or_null(), length(), pos, c);
}

/***************************** rfind methods *****************************/

/** Find substring, backwards. Finds the largest position p such that
    p <= pos and substr(p, str.length()) == str.
    \return p, or npos if no such position exists. */
simple_string::size_type
simple_string::rfind(const simple_string& str, size_type pos) const
{
    return cstring_util::rfind(data_or_null(), length(), pos,
                               str.data_or_null(), str.length());
}

/** Find substring character array, backwards. Finds the largest position p
    such that p <= pos and substr(pos, n) == simple_string(s, n).
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::rfind(const charT* s, size_type pos, size_type n) const
{
    return cstring_util::rfind(data_or_null(), length(), pos, s, n);
}

/** Find C-string. Finds the smallest position p such that
    p <= pos and substr(pos, strlen(s)) == simple_string(s).
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::rfind(const charT* s, size_type pos) const
{
    return cstring_util::rfind(data_or_null(), length(), pos, s, std::strlen(s));
}

/** Find character, backwards. Finds the largest position p such that
    p <= pos and at(p) == c.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::rfind(charT c, size_type pos) const
{
    return cstring_util::rfind(data_or_null(), length(), pos, c);
}

/************************* find_first_of methods *************************/

/** Find first occurence of a character from a set.
    This locates the smallest position p such that p >= pos and
    at(p) occurs in str.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_first_of(const simple_string& str, size_type pos) const
{
    return cstring_util::find_first_of(data_or_null(), length(), pos,
                                       str.data_or_null(), str.length());
}

/** Find first occurence of a character from a set.
    This locates the smallest position p such that p >= pos and
    at(p) occurs in the n-byte array starting at s.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_first_of(const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find_first_of(data_or_null(), length(), pos,
                                       s, n);
}

/** Find first occurence of a character from a set.
    This locates the smallest position p such that p >= pos and
    at(p) occurs in the C-string s.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_first_of(const charT* s, size_type pos) const
{
    return cstring_util::find_first_of(data_or_null(), length(), pos,
                                       s, std::strlen(s));
}

/** Find character. Finds the smallest position p such that p >= pos
    and at(p) == c. This is an alternate name for the respective
    find() signature, provided for convenience.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_first_of(charT c, size_type pos) const
{
    return find(c, pos);
}

/************************** find_last_of methods *************************/

/** Find last occurence of a character from a set.
    This locates the largest position p such that p <= pos and
    at(p) occurs in str.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_last_of (const simple_string& str, size_type pos) const
{
    return cstring_util::find_last_of(data_or_null(), length(), pos,
                                      str.data_or_null(), str.length());
}

/** Find last occurence of a character from a set.
    This locates the largest position p such that p <= pos and
    at(p) occurs in the n-byte array starting at s.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_last_of (const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find_last_of(data_or_null(), length(), pos,
                                      s, n);
}

/** Find last occurence of a character from a set.
    This locates the largest position p such that p <= pos and
    at(p) occurs in the C-string s.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_last_of (const charT* s, size_type pos) const
{
    return cstring_util::find_last_of(data_or_null(), length(), pos,
                                      s, std::strlen(s));
}

/** Find character, backwards. Finds the largest position p such that
    p <= pos and at(p) == c. This is an alternate name for the
    respective rfind() signature, provided for convenience.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_last_of (charT c, size_type pos) const
{
    return rfind(c, pos);
}

/*********************** find_first_not_of methods ***********************/

/** Find first occurence of a character not in a set.
    This locates the smallest position p such that p >= pos and
    at(p) does not occur in str.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_first_not_of(const simple_string& str, size_type pos) const
{
    return cstring_util::find_first_not_of(data_or_null(), length(), pos,
                                           str.data_or_null(), str.length());
}

/** Find first occurence of a character not in a set.
    This locates the smallest position p such that p >= pos and
    at(p) does not occur in the n-byte array starting at s.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_first_not_of(const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find_first_not_of(data_or_null(), length(), pos,
                                           s, n);
}

/** Find first occurence of a character not in a set.
    This locates the smallest position p such that p >= pos and
    at(p) does not occur in the C-string s.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_first_not_of(const charT* s, size_type pos) const
{
    return cstring_util::find_first_not_of(data_or_null(), length(), pos,
                                           s, std::strlen(s));
}


/** Find first non-matching character. This locates the smallest
    position p such that p >= pos and at(p) != c.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_first_not_of(charT c, size_type pos) const
{
    return cstring_util::find_first_not_of(data_or_null(), length(), pos,
                                           &c, 1);
}

/************************ find_last_not_of methods ***********************/

/** Find last occurence of a character not in a set.
    This locates the largest position p such that p <= pos and
    at(p) does not occur in str.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_last_not_of (const simple_string& str, size_type pos) const
{
    return cstring_util::find_last_not_of(data_or_null(), length(), pos,
                                          str.data_or_null(), str.length());
}

/** Find last occurence of a character not in a set.
    This locates the largest position p such that p <= pos and
    at(p) does not occur in the n-byte array s.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_last_not_of (const charT* s, size_type pos, size_type n) const
{
    return cstring_util::find_last_not_of(data_or_null(), length(), pos,
                                          s, n);
}

/** Find last occurence of a character not in a set.
    This locates the largest position p such that p <= pos and
    at(p) does not occur in the C-string s.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_last_not_of (const charT* s, size_type pos) const
{
    return cstring_util::find_last_not_of(data_or_null(), length(), pos,
                                          s, std::strlen(s));
}

/** Find lsat non-matching character. This locates the largest
    position p such that p <= pos and at(p) != c.
    \return p, or npos if no such position exists */
simple_string::size_type
simple_string::find_last_not_of (charT c, size_type pos) const
{
    return cstring_util::find_last_not_of(data_or_null(), length(), pos,
                                          &c, 1);
}

/****************************** misc methods *****************************/

/** Resize to length n. Truncates at the end, or appends c characters
    to enlargen the string. */
void
simple_string::resize(simple_string::size_type n, charT c)
{
    if (n < len)
        assign_from_proxy(proxy(0, txt, n));
    else if (n > len)
        append(n - len, c);
}

/** Reserve memory for future appends. This makes sure that the string
    can reach length n without needing reallocations for these appends.
    \see capacity() */
void
simple_string::reserve(size_type n)
{
    delete ensure_capacity(n);
}

/** Clear this string. This also frees all associated memory.
    \post empty() */
void
simple_string::clear()
{
    if (len || cap) {
        delete[] txt;
        txt = 0;
        cap = len = 0;
    }
}

/** Copy substring to character array. This copies up to \c n
    characters, starting at position \c pos in this string, into
    the character array \c s. The array is not null-terminated.
    \param s character array which receives the output
    \param n number of characters to copy
    \param pos index of first character to copy
    \pre pos <= length()
    \return number of characters copied, n or less. */
simple_string::size_type
simple_string::copy(charT* s, size_type n, size_type pos) const
{
    ASSERT(pos <= length());
    size_type rlen = xstr_min(n, length() - pos);
    std::memcpy(s, data_or_null() + pos, rlen);
    return rlen;
}

/** Checked element access. Obtains read/write access to the character
    at index pos.
    \pre 0 <= pos < length()
    \bug std::string throws an exception, we raise an assertion failure */
simple_string::reference
simple_string::at(size_type pos)
{
    ASSERT(pos < len);
    return txt[pos];
}

/** Checked element access. Obtains read/write access to the character
    at index pos.
    \pre 0 <= pos < length()
    \bug std::string throws an exception, we raise an assertion failure */
simple_string::const_reference
simple_string::at(size_type pos) const
{
    ASSERT(pos < len);
    return txt[pos];
}

/** Substring. Returns a substring of this string.
    \pre 0 <= pos <= length()
    \param pos position of first character to return
    \param n   maximum number of characters to return
    \post return.size() <= n */
simple_string::proxy
simple_string::substr(size_type pos, size_type n) const
{
    ASSERT(pos <= length());
    if (n > length() - pos)
        n = length() - pos;
    if (n)
        return proxy(0, txt + pos, n);
    else
        return proxy(0, 0, 0);
}

/** Swap two strings. This just exchanges the representations,
    therefore it is faster than the naive solution. */
void
simple_string::swap(simple_string& rhs) throw()
{
    charT* txt_save = txt;
    sizeT len_save = len;
    sizeT cap_save = cap;
    txt = rhs.txt;
    len = rhs.len;
    cap = rhs.cap;
    rhs.txt = txt_save;
    rhs.len = len_save;
    rhs.cap = cap_save;
}

simple_string::charT*
simple_string::ensure_capacity(sizeT newcap)
{
    if (newcap <= cap)
        return 0;

    sizeT nc = cap;
    if (nc < SSTRING_ALLOC_MIN)
        nc = SSTRING_ALLOC_MIN;
    while (nc < newcap) {
        if (nc < SSTRING_ALLOC_THRESHOLD)
            nc *= 2;
        else
            nc += SSTRING_ALLOC_INC;
    }
    charT* nb = new char[nc+1];
    charT* rv = txt;
    if (len)
        std::memcpy(nb, txt, len);
    txt = nb;
    cap = nc;
    return rv;
}

/***************************** append methods ****************************/

/** Append another string.
    \return *this */
simple_string&
simple_string::append(const simple_string& str)
{
    append(str.data_or_null(), str.size());
    return *this;
}

/** Append substring. Equivalent to append(str.substr(pos,n)).
    \return *this */
simple_string&
simple_string::append(const simple_string& str, size_type pos, size_type n)
{
    ASSERT(pos <= str.length());

    size_type rest = xstr_min(n, str.length() - pos);
    append(str.data_or_null() + pos, rest);
    return *this;
}

/** Append character array.
    \return *this */
simple_string&
simple_string::append(const charT* s, size_type n)
{
    if (n) {
        charT* dm = ensure_capacity(length() + n);
        std::memcpy(txt + len, s, n);
        len += n;
        delete[] dm;
    }
    return *this;
}

/** Append C string.
    \return *this */
simple_string&
simple_string::append(const charT* s)
{
    append(s, std::strlen(s));
    return *this;
}

/** Append n copies of character c.
    \return *this */
simple_string&
simple_string::append(size_type n, charT c)
{
    charT* dm = ensure_capacity(length() + n);
    std::memset(txt + len, c, n);
    len += n;
    delete[] dm;
    return *this;
}

/***************************** assign methods ****************************/

/** Assign copy of another string.
    \post str == *this
    \return *this */
simple_string&
simple_string::assign(const simple_string& str)
{
    assign_from_proxy(proxy(0, str.data_or_null(), str.length()));
    return *this;
}

/** Assign substring. Equivalent to assign(str.substr(pos, n)).
    \return *this */
simple_string&
simple_string::assign(const simple_string& str, size_type pos,
                      size_type n)
{
    ASSERT(pos <= str.length());

    size_type rest = xstr_min(n, str.length() - pos);
    assign_from_proxy(proxy(0, str.data_or_null() + pos, rest));
    return *this;
}

/** Assign character array.
    \param s character array
    \param n size
    \return *this */
simple_string&
simple_string::assign(const charT* s, size_type n)
{
    assign_from_proxy(proxy(0, s, n));
    return *this;
}

/** Assign C string.
    \return *this */
simple_string&
simple_string::assign(const charT* s)
{
    assign_from_proxy(proxy(0, s, std::strlen(s)));
    return *this;
}

/** Assign n copies of character c.
    \return *this */
simple_string&
simple_string::assign(size_type n, charT c)
{
    charT* v = ensure_capacity(n);
    std::memset(txt, c, n);
    len = n;
    delete[] v;
    return *this;
}

/****************************** operator += ******************************/

/** Assign proxy expression.
    \return *this */
simple_string&
simple_string::operator+=(const proxy& rhs)
{
    return append(simple_string(rhs));
}

/******************************* operator + ******************************/

/** Return concatenation of two strings. */
simple_string::proxy2
operator+(const simple_string& lhs, const simple_string& rhs)
{
    return simple_string::proxy2(0, lhs.data_or_null(), lhs.length(),
                                 rhs.data_or_null(), rhs.length());
}

/** Return concatenation of string expression and a string. */
simple_string::proxy
operator+(const simple_string::proxy& lhs, const simple_string& rhs)
{
    return simple_string::proxy(&lhs, rhs.data_or_null(), rhs.length());
}

/** Return concatenation of a string and a C-string. */
simple_string::proxy2
operator+(const simple_string& lhs, const simple_string::charT* rhs)
{
    return simple_string::proxy2(0, lhs.data_or_null(), lhs.length(),
                                 rhs, std::strlen(rhs));
}

/** Return concatenation of a string expression and a C-string. */
simple_string::proxy
operator+(const simple_string::proxy& lhs, const simple_string::charT* rhs)
{
    return simple_string::proxy(&lhs, rhs, std::strlen(rhs));
}

/** Return concatenation of a C-string and a string. */
simple_string::proxy2
operator+(const simple_string::charT* lhs, const simple_string& rhs)
{
    return simple_string::proxy2(0, lhs, std::strlen(lhs),
                                 rhs.data_or_null(), rhs.length());
}

/** Return concatenation of character and string. */
simple_string
operator+(simple_string::charT lhs, const simple_string& rhs)
{
    return simple_string(1, lhs) + rhs;
}

/** Return concatenation of a string and a character. */
simple_string
operator+(const simple_string& lhs, simple_string::charT rhs)
{
    return lhs + simple_string(1, rhs);
}

/************************** simple_string::proxy *************************/

/** Compute length of a proxy expression. Used to support expressions a la
    (foo + bar).length(). */
simple_string::size_type
simple_string::proxy::length() const
{
    size_type n = 0;
    for (const proxy* p = this; p != 0; p = p->next)
        n += p->len;
    return n;
}
