/**
  *  \file smartptr.cc
  *  \brief Smart Pointer
  *  \author Stefan Reuther
  *
  *  (c) 2001-2005 Stefan Reuther <Streu@gmx.de>
  *
  *  This program is free software; you can redistribute it and/or
  *  modify it under the terms of file `COPYING' that comes with the
  *  source code.
  *
  *  This program is distributed in the hope that it will be useful,
  *  but WITHOUT ANY WARRANTY; without even the implied warranty of
  *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  */
#include "cpluslib/smartptr.h"
#include "cpluslib/assert.h"

#include <map>

#ifdef STREU_CONFIG_DEBUG
/* This attempts to catch illegal assignments of a non-null dumb
   pointer to a Ptr<> during dynamic initialisation or destruction. */
static bool cpluslib_ptr_initialized = false;

struct InitHelper {
    InitHelper();
    ~InitHelper();
};

InitHelper::InitHelper()
{
    cpluslib_ptr_initialized = true;
}

InitHelper::~InitHelper()
{
    cpluslib_ptr_initialized = false;
}

static void
cpluslibEnsurePtrValid()
{
    ASSERT(cpluslib_ptr_initialized);
}

static InitHelper init_watch;
#else
#  define cpluslibEnsurePtrValid()
#endif

static std::map<void*, ReferenceCount*> reference_map;
ReferenceCount ReferenceCount::null_ref;

/** Get reference count for a pointer. This looks up the reference
    count in a map, and allocates a new entry if needed. */
ReferenceCount*
ReferenceCount::getRefCountForPointer(void* p)
{
    if(!p)
        return &ReferenceCount::null_ref;

    cpluslibEnsurePtrValid();

    std::map<void*, ReferenceCount*>::iterator it =
        reference_map.find(p);
    if(it == reference_map.end()) {
        ReferenceCount* rc = new ReferenceCount;
        rc->ptr_count = 0;
        reference_map[p] = rc;
        return rc;
    } else {
        return it->second;
    }
}
