/**
  *  \file holder.cc
  *  \brief Object Lifetime Control
  *
  *  (c) 2001-2005 Stefan Reuther <Streu@gmx.de>
  *
  *  This program is free software; you can redistribute it and/or
  *  modify it under the terms of file `COPYING' that comes with the
  *  source code.
  *
  *  This program is distributed in the hope that it will be useful,
  *  but WITHOUT ANY WARRANTY; without even the implied warranty of
  *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  */
#include "cpluslib/holder.h"

/** Constructor. */
ObjectHolder::ObjectHolder()
{ }

/** Destructor. Deletes all objects in reverse order of add. */
ObjectHolder::~ObjectHolder()
{
    clear();
}

void
ObjectHolder::doAdd(Object* p)
{
    try {
        list.push_back(p);
    }
    catch(...) {
        delete p;
        throw;
    }
}

/** Add object without error handling. This function is to support
    objects which need further processing before they can be deleted. 
    When you're in doubt whether you need this function, you don't
    need it. */
void
ObjectHolder::addNoCatch(Object* p)
{
    list.push_back(p);
}

/** Transfer ownership of our objects to other ObjectHolder. This will
    clear out this ObjectHolder. If the transfer fails, *this is
    cleared in any case, it will never leave this ObjectHolder in a
    partial state. */
void
ObjectHolder::moveTo(ObjectHolder& other)
{
    try {
        for (list_t::iterator i = list.begin(); i != list.end(); ++i) {
            Object* obj = *i;
            other.addNoCatch(obj);
            *i = 0;
        }
    }
    catch (...) {
        // addNoCatch threw. Clear this ObjectHolder to ensure
        // sensible semantics.
        clear();
        throw;
    }
    list.clear();
}

/** Clear this ObjectHolder. Deletes all objects in reverse order of
    addition. */
void
ObjectHolder::clear()
{
    while (!list.empty()) {
        delete list.back(); list.back() = 0;
        list.pop_back();
    }
    list.clear();
}
