/**
  *  \file cstring.cc
  *  \brief Operations on Character Arrays
  *
  *  This file contains a number of C++ string utility functions
  *  suitable for use with raw char arrays (including C strings).
  *  The other string implementations are based on this.
  *
  *  - null-byte clean: all strings are given by pointer+size
  *  - C++ indexing: return values are indexes, npos for not found
  *
  *  (c) 2001-2005 Stefan Reuther <Streu@gmx.de>
  *
  *  This program is free software; you can redistribute it and/or
  *  modify it under the terms of file `COPYING' that comes with the
  *  source code.
  *
  *  This program is distributed in the hope that it will be useful,
  *  but WITHOUT ANY WARRANTY; without even the implied warranty of
  *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  */

#include <cstring>
#include <iostream>
#include "cpluslib/cstring.h"

/** Compare two strings.
    \param astr,alen  string 1
    \param bstr,blen  string 2
    \return zero if equal, negative if astr < bstr, positive if astr > bstr. */
int
cstring_util::compare(charT* astr, sizeT alen,
                      charT* bstr, sizeT blen)
{
    sizeT cmplen = alen > blen ? blen : alen;
    int mres = std::memcmp(astr, bstr, cmplen);
    if (mres)
        return mres;
    else
        return alen > blen ? 1 : alen < blen ? -1 : 0;
}

/** Find substring ("strstr").
    \param haystack,hlen    "haystack" string
    \param hpos             starting position in haystack for search
    \param needle,nlen      "needle" string
    \return position P of needle in haystack such that
    compare(haystack+P,nlen,needle,nlen) == 0 and P>=hpos,
    or npos. */
cstring_util::sizeT
cstring_util::find(charT* haystack, sizeT hlen, sizeT hpos,
                   charT* needle,   sizeT nlen)
{
    if (nlen > hlen || !nlen)
        return npos;

    sizeT maxpos = hlen - nlen;
    while (hpos <= maxpos && (hpos = find(haystack, hlen, hpos, *needle)) != npos && hpos <= maxpos) {
        if (std::memcmp(haystack + hpos, needle, nlen) == 0)
            return hpos;
        ++hpos;
    }
    return npos;
}

/** Find character in string ("strchr").
    \param haystack,hlen    "haystack" string
    \param hpos             starting position for search
    \param needle           character to locate
    \return position P such that haystack[P]==needle && P>=hpos,
    or npos if no such thing. */
cstring_util::sizeT
cstring_util::find(charT* haystack, sizeT hlen, sizeT hpos, char needle)
{
    // gcc doesn't like this one: "static_cast from `void *' to `const char *'"
    //   charT* found = static_cast<charT*>(std::memchr(haystack + hpos, needle, hlen - hpos));
    // I think it's legal (5.2.9p10).
    charT* found = (charT*) std::memchr(haystack + hpos, needle, hlen - hpos);
    return found ? found - haystack : npos;
}

/** Find substring ("strstr" reversed).
    \param haystack,hlen    "haystack" string
    \param hpos             starting position in haystack for search
    \param needle,nlen      "needle" string
    \return position P of needle in haystack such that
    compare(haystack+P,nlen,needle,nlen) == 0 and P<=hpos,
    or npos. */
cstring_util::sizeT
cstring_util::rfind(charT* haystack, sizeT hlen, sizeT hpos,
                    charT* needle,   sizeT nlen)
{
    /* bummer. hpos is the position in "h", but npos isn't
       the position in "n" :-) */
    if (nlen > hlen || !hlen)
        return npos;

    if (hpos > hlen - nlen)
        hpos = hlen - nlen;

    while ((hpos = rfind(haystack, hlen, hpos, *needle)) != npos) {
        if (std::memcmp(haystack + hpos, needle, nlen) == 0)
            return hpos;
        if (!hpos)
            return npos;
        --hpos;
    }
    return npos;
}

/** Find character in string, backwards ("strrchr").
    \param haystack,hlen    "haystack" string
    \param hpos             starting position for search
    \param needle           character to locate
    \return position P such that haystack[P]==needle && P>=hpos,
    or npos if no such thing. */
cstring_util::sizeT
cstring_util::rfind(charT* haystack, sizeT hlen, sizeT hpos, char needle)
{
    if (hpos > hlen)
        hpos = hlen;
    if (hpos < hlen && haystack[hpos] == needle)
        return hpos;
    while (hpos)
        if (haystack[--hpos] == needle)
            return hpos;
    return npos;
}

/** Find first occurence of a set ("strpbrk").
    \param haystack,hlen    "haystack" string
    \param hpos             start at this position
    \param needle,nlen      "needle" string, list of characters
    \return position P such that haystack[P] appears in needle and P>=hpos,
    or npos if no such position. */
cstring_util::sizeT
cstring_util::find_first_of(charT* haystack, sizeT hlen, sizeT hpos,
                            charT* needle,   sizeT nlen)
{
    if (!nlen)
        return npos;

    while (hpos < hlen) {
        if (std::memchr(needle, haystack[hpos], nlen))
            return hpos;
        ++hpos;
    }
    return npos;
}

/** Find last occurence of a set ("strpbrk" reversed).
    \param haystack,hlen    "haystack" string
    \param hpos             start at this position
    \param needle,nlen      "needle" string, list of characters
    \return position P such that haystack[P] appears in needle and P<=hpos,
    or npos if no such position. */
cstring_util::sizeT
cstring_util::find_last_of(charT* haystack, sizeT hlen, sizeT hpos,
                           charT* needle,   sizeT nlen)
{
    if (!nlen || !hlen)
        return npos;

    if (hpos >= hlen)
        hpos = hlen-1;

    while (1) {
        if (std::memchr(needle, haystack[hpos], nlen))
            return hpos;
        if (!hpos--)
            return npos;
    }
}

/** Find last occurence of a set ("strpbrk" inverted).
    \param haystack,hlen    "haystack" string
    \param hpos             start at this position
    \param needle,nlen      "needle" string, list of characters
    \return position P such that haystack[P] does not appear in needle
    and P<=hpos, or npos if no such position. */
cstring_util::sizeT
cstring_util::find_first_not_of(charT* haystack, sizeT hlen, sizeT hpos,
                                charT* needle,   sizeT nlen)
{
    if (!nlen)
        return hpos;            // FIXME?

    while (hpos < hlen) {
        if (!std::memchr(needle, haystack[hpos], nlen))
            return hpos;
        ++hpos;
    }
    return npos;
}

/** Find last occurence of a set ("strpbrk" reversed & inverted).
    \param haystack,hlen    "haystack" string
    \param hpos             start at this position
    \param needle,nlen      "needle" string, list of characters
    \return position P such that haystack[P] does not appear in needle
    and P<=hpos, or npos if no such position. */
cstring_util::sizeT
cstring_util::find_last_not_of(charT* haystack, sizeT hlen, sizeT hpos,
                               charT* needle,   sizeT nlen)
{
    if (!nlen || !hlen)
        return npos;

    if (hpos >= hlen)
        hpos = hlen-1;

    while (1) {
        if (!std::memchr(needle, haystack[hpos], nlen))
            return hpos;
        if (!hpos--)
            return npos;
    }
}
