/**
  *  \file substr.h
  *  \brief Substring
  */
#ifndef STREU_CPLUSLIB_SUBSTR_H_INCLUDED
#define STREU_CPLUSLIB_SUBSTR_H_INCLUDED

#include <iosfwd>
#include "cpluslib/string.h"

/** Substring. This class contains a substring of another string, or a
    statically-allocated C string. This way, it need not manage any
    memory. It behaves like a constant string_t. */
class substring {
 public:
    typedef string_t::value_type       value_type;
    typedef string_t::size_type        size_type;
    typedef string_t::difference_type  difference_type;
    typedef string_t::reference        reference;
    typedef string_t::const_reference  const_reference;
    typedef string_t::pointer          pointer;
    typedef string_t::const_pointer    const_pointer;
    typedef string_t::iterator         iterator;
    typedef string_t::const_iterator   const_iterator;
 private:
    typedef string_t::value_type charT;
    friend class simple_string;

    const charT*  str;
    size_type     len;

    const charT* data_or_null() const { return str; }

 public:
//     typedef std::reverse_iterator<iterator> reverse_iterator;
//     typedef std::reverse_iterator<const_iterator> const_reverse_iterator;

    static const size_type npos = size_type(-1);

    /* constructors. Note that default-constructed substrings
       can't be used until assigned. */
    substring()
        : str(0), len(0) { }
    substring(const charT* src);
    substring(const string_t& rhs)
        : str(rhs.data()), len(rhs.length()) { }
    substring(const string_t& str, size_type pos, size_type n = npos);
    substring(const substring& str, size_type pos, size_type n = npos);
    substring(const charT src[], size_type n);

    size_type    length() const   { return len; }
    const charT* data() const     { return str ? str : ""; }
    size_type    size() const     { return len; }
    size_type    capacity() const { return len; }
    size_type    max_size() const { return npos >> 1; /* be conservative */ }
    void         swap(substring& rhs);
    size_type    copy(charT s[], size_type n, size_type pos = 0) const;

    const_reference operator[](size_type pos) const { return str[pos]; }
    const_reference at(size_type pos) const;

    bool         empty() const { return len==0; }
    void         clear() { len = 0; str = 0; }

    substring& operator=(const substring& rhs) { return assign(rhs); }
    substring& operator=(const string_t& rhs) { return assign(rhs); }
    substring& operator=(const charT* rhs) { return assign(rhs); }

    const_iterator begin() const { return str; }
    const_iterator end()   const { return str + len; }

    substring& assign(const substring& str) { this->str = str.str; this->len = str.len; return *this; }
    substring& assign(const substring& str, size_type pos, size_type n);
    substring& assign(const string_t& str);
    substring& assign(const string_t& str, size_type pos, size_type n);
    substring& assign(const charT* s, size_type n) { str=s; len=n; return *this; }
    substring& assign(const charT* s);

    int compare(const substring& str) const;
    int compare(size_type pos1, size_type n1,
                const substring& str) const;
    int compare(size_type pos1, size_type n1,
                const substring& str,
                size_type pos2, size_type n2) const;
    int compare(const charT* s) const;
    /* compare(pos, n, char_array) != compare(pos, n, string(char_array)) ! */
    int compare(size_type pos1, size_type n1,
                const charT* s, size_type n2 = npos) const;

    substring substr(size_type pos = 0, size_type n = npos) const;

    size_type find (const substring& str, size_type pos = 0) const;
    size_type find (const charT* s, size_type pos, size_type n) const;
    size_type find (const charT* s, size_type pos = 0) const;
    size_type find (charT c, size_type pos = 0) const;

    size_type rfind(const substring& str, size_type pos = npos) const;
    size_type rfind(const charT* s, size_type pos, size_type n) const;
    size_type rfind(const charT* s, size_type pos = npos) const;
    size_type rfind(charT c, size_type pos = npos) const;

    size_type find_first_of(const substring& str, size_type pos = 0) const;
    size_type find_first_of(const charT* s, size_type pos, size_type n) const;
    size_type find_first_of(const charT* s, size_type pos = 0) const;
    size_type find_first_of(charT c, size_type pos = 0) const;

    size_type find_last_of(const substring& str, size_type pos = npos) const;
    size_type find_last_of(const charT* s, size_type pos, size_type n) const;
    size_type find_last_of(const charT* s, size_type pos = npos) const;
    size_type find_last_of(charT c, size_type pos = npos) const;
    
    size_type find_first_not_of(const substring& str, size_type pos = 0) const;
    size_type find_first_not_of(const charT* s, size_type pos, size_type n) const;
    size_type find_first_not_of(const charT* s, size_type pos = 0) const;
    size_type find_first_not_of(charT c, size_type pos = 0) const;

    size_type find_last_not_of(const substring& str, size_type pos = npos) const;
    size_type find_last_not_of(const charT* s, size_type pos, size_type n) const;
    size_type find_last_not_of(const charT* s, size_type pos = npos) const;
    size_type find_last_not_of(charT c, size_type pos = npos) const;

#ifdef STREU_CONFIG_USE_STD_STRING
    /** Convert to C++ string. When string_t is simple_string,
        simple_string has a constructor doing the conversion more
        efficiently, so this one is not needed. */
    operator string_t() const {
        if (length())
            return string_t(data_or_null(), length());
        return string_t();
    }
#endif
};

std::ostream& operator<<(std::ostream&, const substring&);

substring strTrim(substring s);
substring strLTrim(substring s);
substring strRTrim(substring s);
substring strFirst(substring str, substring sep);
bool strRemove(substring& str, const substring sep);
bool strSplit(const substring str, substring& lhs, substring& rhs, const substring sep);

#endif
