/**
  *  \file cpluslib/string.h
  *  \brief String Utilities
  *  \author Stefan Reuther
  *
  *  This file contains a number of useful string operations. In
  *  addition, it defines the string_t typedef which maps to the
  *  preferred string type.
  */
#ifndef STREU_CPLUSLIB_STRING_H_INCLUDED
#define STREU_CPLUSLIB_STRING_H_INCLUDED

#include <iterator>
#include "cpluslib/config.h"

// Definition of string_t

#ifdef STREU_CONFIG_USE_STD_STRING
#  include <string>
typedef std::string string_t;
#else
#  include "cpluslib/sstring.h"
typedef simple_string string_t;
#endif

// Case-insensitive comparison

int strCaseCmp(const char* a, std::size_t na, const char* b, std::size_t nb);
int strCaseCmp(const char* a, const char* b);
int strCaseCmp(const string_t& a, const string_t& b);
int strCaseCmp(const char* a, const string_t& b);
int strCaseCmp(const string_t& a, const char* b);

// Trimming / Parsing

string_t strTrim(string_t s);
string_t strLTrim(string_t s);
string_t strRTrim(string_t s);
string_t strFirst(const string_t str, const string_t sep);
bool     strRemove(string_t& str, const string_t sep);
bool     strSplit(const string_t str, string_t& lhs, string_t& rhs, const string_t sep);

// Case conversion

string_t strLCase(string_t s);
string_t strUCase(string_t s);
string_t strUCFirst(string_t s);
string_t strLCWords(string_t s);

string_t strIndent(const string_t& prefix, string_t str);

string_t itoa(long l);
string_t itoa(long l, const char* a, const char* b);
string_t itoa(long l, const string_t a, const string_t b);

/** String split iterator. Using this iterator, you can treat a string
    split at a given separator as an STL sequence. */
class string_split_iterator {
    typedef std::input_iterator_tag iterator_category;
    typedef std::ptrdiff_t difference_type;
    typedef const string_t* pointer;
    typedef const string_t& reference;
    typedef string_t value_type;
    
    string_t rest;
    string_t split;
    string_t cur;
    bool eof;

    /** Advance to next element. */
    void next()
        {
            if(!rest.length()) {
                eof = true;
                return;
            }
            string_t::size_type n = rest.find(split);
            cur = rest.substr(0, n);
            if(n != string_t::npos)
                n++;
            rest.erase(0, n);
        }
 public:
    /** Construct 'eof' iterator. */
    string_split_iterator()
        : rest(), eof(true) { }
    /** Construct iterator for sequence.
        \param s string to split, e.g. "root:x:0:0:root:/root:/bin/bash"
        \param sp delimiter, e.g. ":" */
    string_split_iterator(string_t s, string_t sp)
        : rest(s), split(sp), eof(false) { next(); }

    /** Get current element. */
    string_t operator*() const
        {
            return cur;
        }

    /** Pre-increment. Advance to next element and return new iterator. */
    string_split_iterator& operator++()
        {
            next();
            return *this;
        }

    /** Post-increment. Advance to next element and return old iterator. */
    const string_split_iterator operator++(int)
        {
            string_split_iterator x = *this;
            next();
            return x;
        }

    /** Compare for equality. */
    bool operator==(const string_split_iterator& rhs)
        {
            return eof == rhs.eof && rest == rhs.rest;
        }

    /** Compare for inequality. */
    bool operator!=(const string_split_iterator& rhs)
        {
            return eof != rhs.eof || rest != rhs.rest;
        }
};

#endif
