/**
  *  \file sstring.h
  *  \brief Small String
  *
  *  A string implementation intended to generate small code.
  */
#ifndef STREU_CPLUSLIB_SSTRING_H_INCLUDED
#define STREU_CPLUSLIB_SSTRING_H_INCLUDED

#include <cstddef>
#include <iosfwd>
#include <algorithm>     /* std::copy */

class substring;

/* Simple string class. */
class simple_string {
    static const char null;
 public:
    class proxy;
    class proxy2;
    class data_proxy;

    friend class substring;

    /* public typedefs etc. */
    typedef char value_type, charT;
    typedef std::size_t size_type, sizeT;
    typedef std::ptrdiff_t difference_type;
    typedef char& reference;
    typedef const char& const_reference;
    typedef char* pointer;
    typedef const char* const_pointer;
 private:
    charT* txt;
    sizeT len;
    sizeT cap;
    void init_from_proxy(const proxy& rhs);
    void assign_from_proxy(const proxy& rhs);
    charT* ensure_capacity(sizeT newcap);

    const char* data_or_null() const { return txt; }
    friend proxy2 operator+(const simple_string&, const simple_string&);
    friend proxy  operator+(const proxy&,         const simple_string&);
    friend proxy2 operator+(const simple_string&, const char*);
    friend proxy2 operator+(const char*,          const simple_string&);
 public:
    typedef char* iterator;
    typedef const char* const_iterator;
//     typedef std::reverse_iterator<iterator> reverse_iterator;
//     typedef std::reverse_iterator<const_iterator> const_reverse_iterator;

    /** Position meaning "not found" or "infinite length". */
    static const size_type npos = size_type(-1);

    /** Constructor. Make an empty string. */
    simple_string()
        : txt(0), len(0), cap(0) { }
    /** Constructor. Initialize from proxy expression. */
    simple_string(const proxy& rhs)
        { init_from_proxy(rhs); }
    simple_string(const charT* src);
    simple_string(const substring& substr);
    simple_string(const simple_string& rhs);
    simple_string(const simple_string& str, size_type pos, size_type n = npos);
    simple_string(const charT* src, size_type n);
    simple_string(size_type n, charT c);

    template<class InputIterator>
    simple_string(InputIterator begin, InputIterator end);

    /** Destructor. */
    ~simple_string()
        { delete[] txt; }

    /** Get length of this string. */
    size_type length() const
        { return len; }
    /** Get data of this string.
        \return pointer to length() bytes making up the string data,
        not necessarily null-terminated. This value is valid until the
        string object is modified. */
    const charT* data() const
        { return txt ? txt : &null; }
    /** Get length of this string. Same as length(). */
    size_type size() const
        { return len; }
    /** Get capacity. The string can grow up to this length without needing
        reallocation. */
    size_type capacity() const
        { return cap; }
    /** Get maximum size this string can reach.
        \bug This is not enforced, to avoid exception handling. */
    size_type max_size() const
        { return npos >> 1; /* be conservative */ }

    void resize(size_type n, charT c);
    /** Resize by appending null characters if needed. */
    void resize(size_type n)
        { resize(n, charT()); }
    void reserve(size_type n = 0);
    void swap(simple_string& rhs) throw();
    size_type copy(charT* s, size_type n, size_type pos = 0) const;

    /** Unchecked access element. Obtains read/write access to element at index pos.
        \pre 0 <= pos < length() */
    reference operator[](size_type pos)
        { return txt[pos]; }
    /** Unchecked access element. Obtains read access to element at index pos.
        If pos == length(), returns the null character.
        \pre 0 <= pos <= length() */
    const_reference operator[](size_type pos) const
        { return *(pos == len ? &null : &txt[pos]); }
    reference at(size_type pos);
    const_reference at(size_type pos) const;

    /** Convert to C string. Returns a zero-terminated, read-only copy
        of this string. This copy is valid until this object is
        modified. */
    const charT* c_str() const
        {
            if (len) {
                txt[len] = 0;
                return txt;
            } else
                return &null;
        }

    /** Check whether this string is empty. */
    bool empty() const
        { return len==0; }
    void clear();

    /** Assign copy of rhs. \return *this */
    simple_string& operator=(const simple_string& rhs)
        { return assign(rhs); }
    /** Assign C-string. \return *this */
    simple_string& operator=(const charT* rhs)
        { return assign(rhs); }
    /** Assign single character.
        \post length() == 1
        \return *this */
    simple_string& operator=(charT rhs)
        { return assign(1, rhs); }
    /** Assign proxy expression. */
    simple_string& operator=(const proxy& rhs)
        {
            assign_from_proxy(rhs);
            return *this;
        }

    /** Append string. \return *this */
    simple_string& operator+=(const simple_string& rhs)
        { return append(rhs); }
    /** Append C-string. \return *this */
    simple_string& operator+=(const charT* rhs)
        { return append(rhs); }
    /** Append character. \return *this */
    simple_string& operator+=(charT rhs)
        { return append(&rhs, 1); }
    simple_string& operator+=(const proxy& rhs);

    /** Get iterator for string beginning. */
    iterator begin()
        { return txt; }
    /** Get constant iterator for string beginning. */
    const_iterator begin() const
        { return txt; }
    /** Get iterator pointing after string end. */
    iterator end()
        { return txt + len; }
    /** Get constant iterator pointing after string end. */
    const_iterator end() const
        { return txt + len; }

    simple_string& append(const simple_string& str);
    simple_string& append(const simple_string& str, size_type pos, size_type n);
    simple_string& append(const charT* s, size_type n);
    simple_string& append(const charT* s);
    simple_string& append(size_type n, charT c);
    template<class InputIterator>
    simple_string& append(InputIterator first, InputIterator last);

    simple_string& assign(const simple_string& str);
    simple_string& assign(const simple_string& str, size_type pos,
                          size_type n);
    simple_string& assign(const charT* s, size_type n);
    simple_string& assign(const charT* s);
    simple_string& assign(size_type n, charT c);
    template<class InputIterator>
    simple_string& assign(InputIterator first, InputIterator last);

    simple_string& erase(size_type pos = 0, size_type n = npos);
    iterator erase(iterator position);
    iterator erase(iterator first, iterator last);

    simple_string& replace(size_type pos1, size_type n1,
                           const simple_string& str);
    simple_string& replace(size_type pos1, size_type n1,
                           const simple_string& str,
                           size_type pos2, size_type n2);
    simple_string& replace(size_type pos, size_type n1, const charT* s, size_type n2);
    simple_string& replace(size_type pos, size_type n1, const charT* s);
    simple_string& replace(size_type pos, size_type n1, size_type n2, charT c);
    simple_string& replace(iterator i1, iterator i2, const simple_string& str);
    simple_string& replace(iterator i1, iterator i2, const charT* s, size_type n);
    simple_string& replace(iterator i1, iterator i2, const charT* s);
    simple_string& replace(iterator i1, iterator i2, size_type n, charT c);

    /** Replace sub-sequence with other sequence.
        \param i1,i2  sequence inside this string to be replaced
        \param j1,j2  sequence to put in instead */
    template<class InputIterator>
    simple_string& replace(iterator i1, iterator i2,
                           InputIterator j1, InputIterator j2)
        { return replace(i1, i2, simple_string(j1, j2)); }

    simple_string& insert(size_type pos1, const simple_string& str);
    simple_string& insert(size_type pos1, const simple_string& str,
                          size_type pos2, size_type n);
    simple_string& insert(size_type pos, const charT* s, size_type n);
    simple_string& insert(size_type pos, const charT* s);
    simple_string& insert(size_type pos, size_type n, charT c);
    iterator insert(iterator p, charT c = charT());
    void insert(iterator p, size_type n, charT c);

    /** Insert sequence at iterator. */
    template<class InputIterator>
    void insert(iterator p, InputIterator first, InputIterator last)
        { insert(p - begin(), simple_string(first, last)); }

    int compare(const simple_string& str) const;
    int compare(size_type pos1, size_type n1,
                const simple_string& str) const;
    int compare(size_type pos1, size_type n1,
                const simple_string& str,
                size_type pos2, size_type n2) const;
    int compare(const charT* s) const;
    /* compare(pos, n, char_array) != compare(pos, n, string(char_array)) ! */
    int compare(size_type pos1, size_type n1,
                const charT* s, size_type n2 = npos) const;

    proxy substr(size_type pos = 0, size_type n = npos) const;

    size_type find (const simple_string& str, size_type pos = 0) const;
    size_type find (const charT* s, size_type pos, size_type n) const;
    size_type find (const charT* s, size_type pos = 0) const;
    size_type find (charT c, size_type pos = 0) const;

    size_type rfind(const simple_string& str, size_type pos = npos) const;
    size_type rfind(const charT* s, size_type pos, size_type n) const;
    size_type rfind(const charT* s, size_type pos = npos) const;
    size_type rfind(charT c, size_type pos = npos) const;

    size_type find_first_of(const simple_string& str, size_type pos = 0) const;
    size_type find_first_of(const charT* s, size_type pos, size_type n) const;
    size_type find_first_of(const charT* s, size_type pos = 0) const;
    size_type find_first_of(charT c, size_type pos = 0) const;

    size_type find_last_of(const simple_string& str,
                           size_type pos = npos) const;
    size_type find_last_of(const charT* s, size_type pos, size_type n) const;
    size_type find_last_of(const charT* s, size_type pos = npos) const;
    size_type find_last_of(charT c, size_type pos = npos) const;

    size_type find_first_not_of(const simple_string& str, size_type pos = 0) const;
    size_type find_first_not_of(const charT* s, size_type pos, size_type n) const;
    size_type find_first_not_of(const charT* s, size_type pos = 0) const;
    size_type find_first_not_of(charT c, size_type pos = 0) const;

    size_type find_last_not_of(const simple_string& str,
                               size_type pos = npos) const;
    size_type find_last_not_of(const charT* s, size_type pos, size_type n) const;
    size_type find_last_not_of(const charT* s, size_type pos = npos) const;
    size_type find_last_not_of(charT c, size_type pos = npos) const;
};

struct simple_string::data_proxy {
    simple_string s;
    data_proxy(const proxy& p)
        : s(p) { }
    operator const charT*() const { return s.c_str(); }
};

struct simple_string::proxy {
    const proxy* next;
    const charT* txt;
    sizeT len;
    proxy(const proxy* an, const charT* ad, sizeT al)
        : next(an), txt(ad), len(al) { }

    /* string primitives */
    /* actually, we should probably also implement the iterator
       primitives, but one can rarely make use of them portably
       anyway, so don't (things like `*begin()' or `end()[-2]'
       are allowed -- but who uses them?). */
    static const size_type npos = size_type(-1);

    size_type length() const;
    data_proxy data() const    { return data_proxy(*this); }
    size_type size() const     { return length(); }
    size_type capacity() const { return length(); }
    size_type max_size() const { return npos >> 1; /* be conservative */ }
    size_type copy(charT* s, size_type n, size_type pos = 0) const
        { return simple_string(*this).copy(s, n, pos); }

    charT operator[](size_type pos) const
        { return simple_string(*this)[pos]; }
    charT at(size_type pos) const
        { return simple_string(*this).at(pos); }

    data_proxy c_str() const { return data_proxy(*this); }
    bool empty() const { return length()==0; } // FIXME: this can be optimized

    int compare(const simple_string& str) const
        // FIXME: these can be optimized
        { return simple_string(*this).compare(str); }
    int compare(size_type pos1, size_type n1, const simple_string& str) const
        { return simple_string(*this).compare(pos1, n1, str); }
    int compare(size_type pos1, size_type n1, const simple_string& str,
                size_type pos2, size_type n2) const
        { return simple_string(*this).compare(pos1, n1, str, pos2, n2); }
    int compare(const charT* s) const
        { return simple_string(*this).compare(s); }
    int compare(size_type pos1, size_type n1,
                const charT* s, size_type n2 = npos) const
        { return simple_string(*this).compare(pos1, n1, s, n2); }

    simple_string substr(size_type pos = 0, size_type n = npos) const
        // FIXME: this can be optimized to return a proxy
        { return simple_string(*this).substr(pos, n); }

    size_type find (const simple_string& str, size_type pos = 0) const
        { return simple_string(*this).find(str, pos); }
    size_type find (const charT* s, size_type pos, size_type n) const
        { return simple_string(*this).find(s, pos, n); }
    size_type find (const charT* s, size_type pos = 0) const
        { return simple_string(*this).find(s, pos); }
    size_type find (charT c, size_type pos = 0) const
        { return simple_string(*this).find(c, pos); }

    size_type rfind(const simple_string& str, size_type pos = npos) const
        { return simple_string(*this).rfind(str, pos); }
    size_type rfind(const charT* s, size_type pos, size_type n) const
        { return simple_string(*this).rfind(s, pos, n); }
    size_type rfind(const charT* s, size_type pos = npos) const
        { return simple_string(*this).rfind(s, pos); }
    size_type rfind(charT c, size_type pos = npos) const
        { return simple_string(*this).rfind(c, pos); }

    size_type find_first_of(const simple_string& str, size_type pos = 0) const
        { return simple_string(*this).find_first_of(str, pos); }
    size_type find_first_of(const charT* s, size_type pos, size_type n) const
        { return simple_string(*this).find_first_of(s, pos, n); }
    size_type find_first_of(const charT* s, size_type pos = 0) const
        { return simple_string(*this).find_first_of(s, pos); }
    size_type find_first_of(charT c, size_type pos = 0) const
        { return simple_string(*this).find_first_of(c, pos); }

    size_type find_last_of(const simple_string& str, size_type pos = npos) const
        { return simple_string(*this).find_last_of(str, pos); }
    size_type find_last_of(const charT* s, size_type pos, size_type n) const
        { return simple_string(*this).find_last_of(s, pos, n); }
    size_type find_last_of(const charT* s, size_type pos = npos) const
        { return simple_string(*this).find_last_of(s, pos); }
    size_type find_last_of(charT c, size_type pos = npos) const
        { return simple_string(*this).find_last_of(c, pos); }

    size_type find_first_not_of(const simple_string& str, size_type pos = 0) const
        { return simple_string(*this).find_first_not_of(str, pos); }
    size_type find_first_not_of(const charT* s, size_type pos, size_type n) const
        { return simple_string(*this).find_first_not_of(s, pos, n); }
    size_type find_first_not_of(const charT* s, size_type pos = 0) const
        { return simple_string(*this).find_first_not_of(s, pos); }
    size_type find_first_not_of(charT c, size_type pos = 0) const
        { return simple_string(*this).find_first_not_of(c, pos); }

    size_type find_last_not_of(const simple_string& str, size_type pos = npos) const
        { return simple_string(*this).find_last_not_of(str, pos); }
    size_type find_last_not_of(const charT* s, size_type pos, size_type n) const
        { return simple_string(*this).find_last_not_of(s, pos, n); }
    size_type find_last_not_of(const charT* s, size_type pos = npos) const
        { return simple_string(*this).find_last_not_of(s, pos); }
    size_type find_last_not_of(charT c, size_type pos = npos) const
        { return simple_string(*this).find_last_not_of(c, pos); }
};

struct simple_string::proxy2 : simple_string::proxy {
    proxy p2;
    proxy2(const proxy* an, const charT* ad, sizeT al,
           const charT* a2, sizeT l2)
        : proxy(&p2, a2, l2), p2(an, ad, al) { }
};

/************************** comparison operators *************************/

inline bool operator<(const simple_string& lhs,        const simple_string& rhs)        { return lhs.compare(rhs) < 0; }
inline bool operator<(const simple_string& lhs,        const simple_string::charT* rhs) { return lhs.compare(rhs) < 0; }
inline bool operator<(const simple_string::charT* lhs, const simple_string& rhs)        { return rhs.compare(lhs) > 0; }

inline bool operator>(const simple_string& lhs,        const simple_string& rhs)        { return lhs.compare(rhs) > 0; }
inline bool operator>(const simple_string& lhs,        const simple_string::charT* rhs) { return lhs.compare(rhs) > 0; }
inline bool operator>(const simple_string::charT* lhs, const simple_string& rhs)        { return rhs.compare(lhs) < 0; }

inline bool operator<=(const simple_string& lhs,        const simple_string& rhs)        { return lhs.compare(rhs) <= 0; }
inline bool operator<=(const simple_string& lhs,        const simple_string::charT* rhs) { return lhs.compare(rhs) <= 0; }
inline bool operator<=(const simple_string::charT* lhs, const simple_string& rhs)        { return rhs.compare(lhs) >= 0; }

inline bool operator>=(const simple_string& lhs,        const simple_string& rhs)        { return lhs.compare(rhs) >= 0; }
inline bool operator>=(const simple_string& lhs,        const simple_string::charT* rhs) { return lhs.compare(rhs) >= 0; }
inline bool operator>=(const simple_string::charT* lhs, const simple_string& rhs)        { return rhs.compare(lhs) <= 0; }

inline bool operator==(const simple_string& lhs,        const simple_string& rhs)        { return lhs.compare(rhs) == 0; }
inline bool operator==(const simple_string& lhs,        const simple_string::charT* rhs) { return lhs.compare(rhs) == 0; }
inline bool operator==(const simple_string::charT* lhs, const simple_string& rhs)        { return rhs.compare(lhs) == 0; }

inline bool operator!=(const simple_string& lhs,        const simple_string& rhs)        { return lhs.compare(rhs) != 0; }
inline bool operator!=(const simple_string& lhs,        const simple_string::charT* rhs) { return lhs.compare(rhs) != 0; }
inline bool operator!=(const simple_string::charT* lhs, const simple_string& rhs)        { return rhs.compare(lhs) != 0; }

/******************************* Templates *******************************/

/** Construct string from a sequence. */
template<class InputIterator>
simple_string::simple_string(InputIterator begin, InputIterator end)
{
    len = std::distance(begin, end);
    txt = new charT[len+1];
    std::copy(begin, end, txt);
}

/** Append sequence.
    \return *this */
template<class InputIterator>
simple_string&
simple_string::append(InputIterator first, InputIterator last)
{
    append(simple_string(first, last));
    return *this;
}

/** Assign sequence.
    \return *this */
template<class InputIterator>
simple_string&
simple_string::assign(InputIterator first, InputIterator last)
{
    assign(simple_string(first, last));
    return *this;
}

/************************** Operator Prototypes **************************/

simple_string::proxy2 operator+(const simple_string& lhs, const simple_string& rhs);
simple_string::proxy  operator+(const simple_string::proxy& lhs, const simple_string& rhs);
simple_string::proxy2 operator+(const simple_string& lhs, const simple_string::charT* rhs);
simple_string::proxy  operator+(const simple_string::proxy& lhs, const simple_string::charT* rhs);
simple_string::proxy2 operator+(const simple_string::charT* lhs, const simple_string& rhs);
simple_string         operator+(simple_string::charT lhs, const simple_string& rhs);
simple_string         operator+(const simple_string& lhs, simple_string::charT rhs);

std::ostream& operator<<(std::ostream& os, const simple_string& str);

/******************************* Functions *******************************/

/** Swap two strings. */
inline void
swap(simple_string& lhs, simple_string& rhs)
{
    lhs.swap(rhs);
}

std::istream& getline(std::istream& is, simple_string& ss);
std::istream& getline(std::istream& is, simple_string& ss, char delim);

#endif
