/**
  *  \file cpluslib/smartptr.h
  *  \brief Smart Pointer
  *  \author Stefan Reuther
  */
#ifndef STREU_CPLUSLIB_SMARTPTR_H_INCLUDED
#define STREU_CPLUSLIB_SMARTPTR_H_INCLUDED

#include "cpluslib/config.h"
#include "cpluslib/object.h"
#include <cstdlib>

/** Reference-counted pointer. This pointer supports all operations
    which a dumb pointer-to-object would support. In addition, it
    counts the number of active pointers to one object, if the last
    pointer dies, it deletes the object. To point to an array, use
    ArrayPtr.

    Ptr<> exports its representation to allow other, more efficient,
    code to interface it. In particular, it is possible to put
    Ptr<>-managed pointers into unions if you manually manage the
    reference counts.

    It is possible to pass a smart pointer "through" a dumb one. It
    isn't particularly efficient, though.

    Restrictions:
    - reference-counted pointers do not support circular structures
    - you cannot initialize a namespace-scope Ptr<T> to something
      other than 0, because the reference count management might not
      have been set up by that time.
    - allocated reference count objects are never freed.
    - this requires that, if d is a Derived*, (void*)(Base*) d == (void*) d.
      Although not required by ISO C++, this is true in all
      compilers known to me (GNU, Borland, MS, ADI) for single
      inheritance, and never true for multiple inheritance. If this
      identity does not hold, you can no longer pass a smart pointer
      throug a dumb one, but intra-Ptr<> copying still works.
      Essentially, this rules out all sorts of multiple inheritance.

    Many restrictions are lifted if the pointee is derived from Object
    and the optimisation (STREU_CONFIG_PTR_OPTIMIZE_OBJECT) is
    enabled. In this case
    - smart-pointer-through-dumb-one is fast
    - namespace-scope Ptr<> is allowed
    - reference count is freed with the object
    - Ptr<> is guaranteed to work for single inheritance even if the
      above identity does not hold.
    - if the derivation from Object is virtual (i.e. there's only one
      Object base-class sub-object), multiple inheritance is supported,
      too.

    With debugging enabled, Ptr<> attempts to make sure you do not
    create non-null pointers at namespace scope. */
template<class T>
class Ptr {
 public:
    /** Representation. This one's exported for various reasons. In
        particular, Ptr<T>::Ref is POD, so it can be put in a union. 
        Just be sure to update the reference counts accordingly... */
    struct Ref {
        T* ptr;
        ReferenceCount* rc;
    };
    Ref ref;
    /** Decrement reference count. */
    void dec() const
        {
            if(!--ref.rc->ptr_count)
                delete ref.ptr;
        }
    /** Increment reference count. */
    void inc() const
        { ++ref.rc->ptr_count; }
 public:
    /** Construct null pointer. */
    Ptr()
        {
            ref.ptr = 0;
            ref.rc = &ReferenceCount::null_ref;
            ++ref.rc->ptr_count;
        }
    /** Construct pointer (to object or null). */
    Ptr(T* ap)
        {
            ref.ptr = ap;
            ref.rc = ReferenceCount::getRefCountForPointer(ap);
            inc();
        }
    /** Copy constructor. */
    Ptr(const Ptr& a)
        : ref(a.ref)
        { inc(); }
    /** Converting copy constructor. */
    template<class T1>
    Ptr(const Ptr<T1>& a)
        {
            ref.ptr = a.ref.ptr;
            ref.rc  = a.ref.rc;
            inc();
        }

    /** Destructor. */
    ~Ptr()
        { dec(); };

    /** Assignment operator. */
    Ptr& operator=(const Ptr& a)
        {
            Ref r = a.ref;
            a.inc();
            dec();
            ref = r;
            return *this;
        }
    /** Converting assignment operator. */
    template<class T1>
    Ptr& operator=(const Ptr<T1>& a)
        {
            Ref r;
            r.ptr = a.ref.ptr;
            r.rc  = a.ref.rc;
            a.inc();
            dec();
            ref = r;
            return *this;
        }
    /** Assign dumb pointer. */
    Ptr& operator=(T* a)
        {
            ReferenceCount* new_ref = ReferenceCount::getRefCountForPointer(a);
            ++new_ref->ptr_count;
            dec();
            ref.rc = new_ref;
            ref.ptr = a;
            return *this;
        }

    /** Extract dumb pointer. Use this to make the dumb-pointer access
        explicit. */
    T* ptr() const { return ref.ptr; }

    /** Get reference count. */
    std::size_t getReferenceCount() const
        { return ref.rc->ptr_count; }

    /** Convert to dumb pointer. */
    operator T*const() const { return ref.ptr; }

    T& operator*() const
        { return *ref.ptr; }
    T* operator->() const
        { return ref.ptr; }

    /** Smart dynamic_cast.
        \todo take advantage of the fact that the result is either 0 or ptr(). */
    template<class T1>
    Ptr<T1> cast() const
        { return dynamic_cast<T1*>(this->ptr()); }
};

/** Reference-counted pointer to an array. This class is mostly
    identical to Ptr<>, see there for more details. However, it
    manages a pointer to an array, not to a single object.

    The optimisations for pointers to Object, and the improved
    behaviour, does not apply to ArrayPtr. */
template<class T>
class ArrayPtr {
 public:
    struct Ref {
        T* ptr;
        ReferenceCount* rc;
        Ref(T* aptr, ReferenceCount* arc) : ptr(aptr), rc(arc) { }
        Ref(T* aptr) : ptr(aptr), rc(ReferenceCount::getRefCountForArray(aptr)) { }
    };
    Ref ref;
    /** Decrement reference count. */
    void dec() const
        {
            if(!--ref.rc->ptr_count) {
                delete[] ref.ptr;
            }
        }
    /** Increment reference count. */
    void inc() const
        { ++ref.rc->ptr_count; }
 public:
    /** Constructor. */
    ArrayPtr(T* ap = 0)
        : ref(ap, ReferenceCount::getRefCountForArray(ap))
        { inc(); }
    /** Copy constructor. */
    ArrayPtr(const ArrayPtr& a)
        : ref(a.ref)
        { inc(); }
    /** Create new array. */
    explicit ArrayPtr(std::size_t size)
        : ref(new T[size])
        { inc(); }
    /** Destructor. */
    ~ArrayPtr()
        { dec(); };

    /** Assignment operator. */
    ArrayPtr& operator=(const ArrayPtr& a)
        {
            Ref r = a.ref;
            a.inc();
            dec();
            ref = r;
            return *this;
        }
    /** Assign dumb array pointer. */
    ArrayPtr& operator=(T* a)
        {
            ReferenceCount* new_ref = ReferenceCount::getRefCountForArray(a);
            ++new_ref->ptr_count;
            dec();
            ref.rc = new_ref;
            ref.ptr = a;
            return *this;
        }
    /** Get dumb pointer. */
    T* ptr() const
        { return ref.ptr; }
    /** Get reference count. */
    std::size_t getReferenceCount() const
        { return ref.rc->ptr_count; }

    T& operator*() const { return *ref.ptr; }
    T* operator->() const { return ref.ptr; }

    /** Element access. */
    T& operator[](std::size_t ix) const
        { return ref.ptr[ix]; }
};

#endif
