/**
  *  \file smallset.h
  *  \brief Small Set
  *  \author Stefan Reuther
  *
  *  This header contains the SmallSet template class, which
  *  implements a type-safe wrapper for set operations with unsigned
  *  integer types.
  */
#ifndef STREU_CPLUSLIB_SMALLSET_H_INCLUDED
#define STREU_CPLUSLIB_SMALLSET_H_INCLUDED

/** Small set. This encapsulates a type-safe set where members are
    enumerators with numeric values corresponding to the bits of an
    unsigned long (that is, 0..31, usually).

    \param T    Member type
    \param Rep  Underlying type. Must be an integer. */
template<typename T, typename Rep = unsigned long>
class SmallSet {
    Rep rep;

    SmallSet(Rep l, bool)
        : rep(l) { }
 public:
    /** Underlying type. */
    typedef Rep rep_t;

    /** Value type. This is the type contained in this set. */
    typedef T value_type;
    
    /** Construct empty set. */
    SmallSet()
        : rep(0) { }

    /** Construct unit set. */
    explicit
    SmallSet(T t)
        : rep(rep_t(1) << t) { }

    /** Add an element. */
    SmallSet&
    operator |= (T t)
        { rep |= (rep_t(1) << t); return *this; }

    /** Add a set. */
    SmallSet&
    operator |= (SmallSet other)
        { rep |= other.rep; return *this; }

    /** Set union (set plus single element). */
    SmallSet
    operator | (T t) const
        { return SmallSet<T>(*this) |= t; }

    /** Set union. */
    SmallSet
    operator | (SmallSet other) const
        { return SmallSet<T>(*this) |= other; }

    /** Add an element (alternate name). */
    SmallSet&
    operator += (T t)
        { rep |= (rep_t(1) << t); return *this; }

    /** Add a set (alternate name). */
    SmallSet&
    operator += (SmallSet other)
        { rep |= other.rep; return *this; }

    /** Set union (set plus single element, alternate name). */
    SmallSet
    operator + (T t) const
        { return SmallSet<T>(*this) |= t; }

    /** Set union (alternate name). */
    SmallSet
    operator + (SmallSet other) const
        { return SmallSet<T>(*this) |= other; }

    /** Remove an element. */
    SmallSet&
    operator -= (T t)
        { rep &= ~(rep_t(1) << t); return *this; }

    /** Remove a set. */
    SmallSet&
    operator -= (SmallSet other)
        { rep &= ~other.rep; return *this; }

    /** Set difference (set minus single element). */
    SmallSet
    operator - (T t) const
        { return SmallSet<T>(*this) -= t; }

    /** Set difference. */
    SmallSet
    operator - (SmallSet other) const
        { return SmallSet<T>(*this) -= other; }

    /** Conjunction (intersection). */
    SmallSet&
    operator &= (SmallSet other)
        { rep &= other.rep; return *this; }

    /** Conjuction (intersection). */
    SmallSet
    operator & (SmallSet other) const
        { return SmallSet<T>(*this) &= other; }

    /** Toggle an element (symmetric difference). */
    SmallSet&
    operator ^= (T t)
        { rep ^= (rep_t(1) << t); return *this; }

    /** Toggle elements from other set (symmetric difference). */
    SmallSet&
    operator ^= (SmallSet other)
        { rep ^= other.rep; return *this; }

    /** Symmetric set difference. */
    SmallSet
    operator ^ (T t) const
        { return SmallSet<T>(*this) ^= t; }

    /** Symmetric set difference. */
    SmallSet
    operator ^ (SmallSet other) const
        { return SmallSet<T>(*this) ^= other; }

    /** Test whether element is in. */
    bool
    contains(T t) const
        { return rep & (rep_t(1) << t); }

    /** Test whether this set contains all elements from the other one. */
    bool
    contains(SmallSet other) const
        { return (other.rep & ~rep) == 0; }

    /** Test whether this set contains any of the elements from the
        other one. */
    bool
    containsAnyOf(SmallSet other) const
        { return (rep & other.rep) != 0; }

    /** Test whether this set is empty. */
    bool
    empty() const
        { return rep == 0; }

    /** Clear this set. \post empty() */
    void
    clear()
        { rep = 0; }

    /** Check whether this set contains at most one element. */
    bool
    isUnitSet() const
        { return (rep & (rep-1)) == 0; }

    /** Static constructor: construct a set which contains all objects
        from T(0) up to T(t) */
    static SmallSet
    allUpTo(T t)
        {
            // this relies on the guaranteed overflow behaviour
            return SmallSet<T>((rep_t(1) << (t+1)) - 1, true);
        }

    /** Compare for equality. */
    bool
    operator==(SmallSet other) const
        {
            return rep == other.rep;
        }

    /** Compare for inequality. */
    bool
    operator!=(SmallSet other) const
        {
            return rep != other.rep;
        }

    /** True iff set is non-empty. Shortcut for !empty(). */
    bool
    nonempty() const
        {
            return rep != 0;
        }

    /** Get representation. Returns an integer where
        a.contains(x) iff (a.toInteger() & (1 << x)) != 0 */
    rep_t toInteger() const
        { return rep; }

    /** Create set from integer. Each set bit in \c v translates into
        a set element. */
    static SmallSet fromInteger(rep_t v)
        { return SmallSet<T>(v, true); }
};

#endif
